class ScrapingManager {
    constructor() {
        this.isRunning = false;
        this.isPaused = false;
        this.currentSitemap = null;
        this.scrapedData = [];
        this.stats = {
            records: 0,
            pages: 0,
            duplicates: 0,
            errors: 0,
            startTime: null,
            endTime: null
        };
        this.activeTabId = null;
        this.currentPageIndex = 0;
        this.visitedUrls = new Set();
        this.recordHashes = new Set();
        this.runControls = {
            shouldStop: false,
            shouldPause: false
        };
        
        this.initializeListeners();
        this.loadState();
    }

    initializeListeners() {
        // Handle messages from popup and content scripts
        chrome.runtime.onMessage.addListener((message, sender, sendResponse) => {
            this.handleMessage(message, sender, sendResponse);
            return true; // Keep message channel open for async responses
        });

        // Handle tab updates
        chrome.tabs.onUpdated.addListener((tabId, changeInfo, tab) => {
            if (this.isRunning && tabId === this.activeTabId && changeInfo.status === 'complete') {
                this.handlePageLoaded(tab);
            }
        });

        // Handle extension startup
        chrome.runtime.onStartup.addListener(() => {
            this.loadState();
        });

        // Handle extension installation
        chrome.runtime.onInstalled.addListener(() => {
            this.initializeStorage();
        });
    }

    async initializeStorage() {
        const defaultData = {
            sitemaps: [],
            scrapedData: [],
            scrapingStats: this.stats,
            scrapingState: { isRunning: false },
            settings: {
                rateLimit: 1000, // ms between requests
                maxPages: 100,
                infiniteScrollDelay: 2000,
                duplicateDetection: true
            }
        };

        await chrome.storage.local.set(defaultData);
    }

    async loadState() {
        try {
            const result = await chrome.storage.local.get([
                'scrapingState',
                'currentSitemap',
                'scrapedData',
                'scrapingStats'
            ]);

            this.isRunning = result.scrapingState?.isRunning || false;
            this.currentSitemap = result.currentSitemap || null;
            this.scrapedData = result.scrapedData || [];
            this.stats = result.scrapingStats || this.stats;
        } catch (error) {
            console.error('Error loading state:', error);
        }
    }

    async saveState() {
        try {
            await chrome.storage.local.set({
                scrapingState: { isRunning: this.isRunning },
                currentSitemap: this.currentSitemap,
                scrapedData: this.scrapedData,
                scrapingStats: this.stats
            });
        } catch (error) {
            console.error('Error saving state:', error);
        }
    }

    async handleMessage(message, sender, sendResponse) {
        try {
            const messageType = message.type || message.action; // Support both for compatibility
            switch (messageType) {
                case 'startScraping':
                    await this.startScraping(message.sitemap);
                    sendResponse({ success: true });
                    break;

                case 'pauseScraping':
                    await this.pauseScraping();
                    sendResponse({ success: true });
                    break;

                case 'resumeScraping':
                    await this.resumeScraping();
                    sendResponse({ success: true });
                    break;

                case 'stopScraping':
                    await this.stopScraping();
                    sendResponse({ success: true });
                    break;
                    
                case 'getStats':
                    sendResponse({ stats: this.getRealtimeStats(), isRunning: this.isRunning, isPaused: this.isPaused });
                    break;
                    
                case 'scrollProgress':
                    this.handleScrollProgress(message.progress);
                    sendResponse({ success: true });
                    break;

                case 'exportData':
                    try {
                        const result = await chrome.storage.local.get(['scrapedData', 'currentSitemap']);
                        const data = result.scrapedData || [];
                        const sitemap = result.currentSitemap;
                        const format = message.format || 'csv';
                        const filters = message.filters || {};
                        
                        if (data.length === 0) {
                            sendResponse({ success: false, error: 'No data to export' });
                            return;
                        }
                        
                        // Apply filters if provided
                        let filteredData = data;
                        
                        if (filters.sitemapId) {
                            filteredData = filteredData.filter(item => item._sitemapId === filters.sitemapId);
                        }
                        
                        if (filters.dateFrom || filters.dateTo) {
                            const startDate = filters.dateFrom ? new Date(filters.dateFrom) : null;
                            const endDate = filters.dateTo ? new Date(filters.dateTo) : null;
                            
                            filteredData = filteredData.filter(item => {
                                const itemDate = new Date(item._scrapedAt);
                                if (startDate && itemDate < startDate) return false;
                                if (endDate && itemDate > endDate) return false;
                                return true;
                            });
                        }
                        
                        if (filteredData.length === 0) {
                            sendResponse({ success: false, error: 'No data matches the selected filters' });
                            return;
                        }
                        
                        // Generate filename
                        const timestamp = new Date().toISOString().replace(/[:.]/g, '-').slice(0, -5);
                        const sitemapName = sitemap?.name || 'data';
                        const safeName = sitemapName.replace(/[^a-zA-Z0-9]/g, '-').toLowerCase();
                        const filename = `ai-spun-${safeName}-${timestamp}`;
                        
                        let blob, downloadFilename;
                        
                        if (format === 'xlsx') {
                            // Create XLSX content using simplified format
                            const headers = Object.keys(filteredData[0]);
                            const xlsxContent = this.createSimpleXLSX(filteredData, headers);
                            blob = new Blob([xlsxContent], { 
                                type: 'application/vnd.openxmlformats-officedocument.spreadsheetml.sheet' 
                            });
                            downloadFilename = `${filename}.xlsx`;
                        } else {
                            // Create CSV content
                            const headers = Object.keys(filteredData[0]);
                            const csvContent = this.arrayToCSV(filteredData, headers);
                            blob = new Blob([csvContent], { type: 'text/csv;charset=utf-8;' });
                            downloadFilename = `${filename}.csv`;
                        }
                        
                        // Create download
                        const url = URL.createObjectURL(blob);
                        
                        await chrome.downloads.download({
                            url: url,
                            filename: downloadFilename,
                            saveAs: true
                        });
                        
                        // Clean up the URL object after a delay
                        setTimeout(() => URL.revokeObjectURL(url), 1000);
                        
                        sendResponse({ 
                            success: true, 
                            recordsExported: filteredData.length,
                            format: format 
                        });
                    } catch (error) {
                        console.error('Export error:', error);
                        sendResponse({ success: false, error: error.message });
                    }
                    break;

                case 'getSitemaps':
                    const sitemaps = await this.getSitemaps();
                    sendResponse({ sitemaps });
                    break;

                case 'saveSitemap':
                    await this.saveSitemap(message.sitemap);
                    sendResponse({ success: true });
                    break;

                case 'deleteSitemap':
                    await this.deleteSitemap(message.sitemapId);
                    sendResponse({ success: true });
                    break;

                case 'getScrapedData':
                    sendResponse({ data: this.scrapedData });
                    break;

                case 'clearData':
                    await this.clearData();
                    sendResponse({ success: true });
                    break;

                case 'dataScraped':
                    await this.handleDataScraped(message.data, sender.tab.id);
                    sendResponse({ success: true });
                    break;

                case 'pageComplete':
                    await this.handlePageComplete(sender.tab.id);
                    sendResponse({ success: true });
                    break;

                default:
                    sendResponse({ error: 'Unknown action' });
            }
        } catch (error) {
            console.error('Error handling message:', error);
            sendResponse({ error: error.message });
        }
    }

    async startScraping(sitemap) {
        if (this.isRunning) {
            throw new Error('Scraping is already running');
        }

        this.currentSitemap = sitemap;
        this.isRunning = true;
        this.isPaused = false;
        this.currentPageIndex = 0;
        this.visitedUrls.clear();
        this.recordHashes.clear();
        
        // Reset run controls
        this.runControls = {
            shouldStop: false,
            shouldPause: false
        };
        
        // Reset stats for new scraping session
        this.stats = {
            records: 0,
            pages: 0,
            duplicates: 0,
            errors: 0,
            startTime: Date.now(),
            endTime: null
        };
        
        await this.saveState();
        this.broadcastMessage({ action: 'scrapingStarted', stats: this.getRealtimeStats() });

        // Start scraping from first URL
        if (sitemap.startUrls && sitemap.startUrls.length > 0) {
            await this.navigateToUrl(sitemap.startUrls[0]);
        }
    }

    async pauseScraping() {
        this.isPaused = true;
        this.runControls.shouldPause = true;
        await this.saveState();
        this.broadcastMessage({ action: 'scrapingPaused' });
    }
    
    async resumeScraping() {
        this.isPaused = false;
        this.runControls.shouldPause = false;
        await this.saveState();
        this.broadcastMessage({ action: 'scrapingResumed' });
    }
    
    async stopScraping() {
        this.isRunning = false;
        this.isPaused = false;
        this.runControls.shouldStop = true;
        this.stats.endTime = Date.now();
        await this.saveState();
        this.broadcastMessage({ action: 'scrapingStopped', stats: this.getRealtimeStats() });
    }
    
    getRealtimeStats() {
        const runtime = this.stats.startTime ? 
            (this.stats.endTime || Date.now()) - this.stats.startTime : 0;
        
        return {
            ...this.stats,
            runtime: runtime,
            runtimeFormatted: this.formatRuntime(runtime),
            recordsPerMinute: runtime > 0 ? Math.round((this.stats.records / runtime) * 60000) : 0,
            isRunning: this.isRunning,
            isPaused: this.isPaused
        };
    }
    
    formatRuntime(ms) {
        const seconds = Math.floor(ms / 1000);
        const minutes = Math.floor(seconds / 60);
        const hours = Math.floor(minutes / 60);
        
        if (hours > 0) {
            return `${hours}h ${minutes % 60}m ${seconds % 60}s`;
        } else if (minutes > 0) {
            return `${minutes}m ${seconds % 60}s`;
        } else {
            return `${seconds}s`;
        }
    }
    
    handleScrollProgress(progress) {
        // Broadcast scroll progress to popup for real-time updates
        this.broadcastMessage({ 
            action: 'scrollProgress', 
            progress: progress 
        });
     }

    async navigateToUrl(url) {
        if (this.visitedUrls.has(url)) {
            return; // Skip already visited URLs
        }

        this.visitedUrls.add(url);
        
        try {
            // Create or update tab
            if (this.activeTabId) {
                await chrome.tabs.update(this.activeTabId, { url });
            } else {
                const tab = await chrome.tabs.create({ url, active: false });
                this.activeTabId = tab.id;
            }
        } catch (error) {
            console.error('Error navigating to URL:', error);
            this.broadcastMessage({ type: 'error', message: `Failed to navigate to ${url}` });
        }
    }

    async handlePageLoaded(tab) {
        if (!this.isRunning || !this.currentSitemap) {
            return;
        }

        this.stats.pages++;
        await this.saveState();
        this.broadcastMessage({ type: 'statsUpdated', stats: this.stats });

        // Inject content script and start scraping
        try {
            await chrome.tabs.sendMessage(tab.id, {
                action: 'startPageScraping',
                sitemap: this.currentSitemap
            });
        } catch (error) {
            console.error('Error sending message to content script:', error);
        }
    }

    async handleDataScraped(data, tabId) {
        if (!Array.isArray(data)) {
            data = [data];
        }

        let newRecords = 0;
        let duplicates = 0;

        for (const record of data) {
            // Simple duplicate detection based on JSON string comparison
            const recordString = JSON.stringify(record);
            const isDuplicate = this.scrapedData.some(existing => 
                JSON.stringify(existing) === recordString
            );

            if (isDuplicate) {
                duplicates++;
            } else {
                this.scrapedData.push({
                    ...record,
                    _scrapedAt: new Date().toISOString(),
                    _sourceUrl: await this.getTabUrl(tabId)
                });
                newRecords++;
            }
        }

        this.stats.records += newRecords;
        this.stats.duplicates += duplicates;

        await this.saveState();
        this.broadcastMessage({ type: 'statsUpdated', stats: this.stats });

        // Auto-save every 50 records
        if (this.stats.records % 50 === 0) {
            await this.autoSave();
        }
    }

    async handlePageComplete(tabId) {
        if (!this.isRunning) {
            return;
        }

        // Check for pagination or next URL
        const hasNextPage = await this.checkForNextPage(tabId);
        
        if (!hasNextPage) {
            // Check if there are more start URLs to process
            if (this.currentPageIndex < this.currentSitemap.startUrls.length - 1) {
                this.currentPageIndex++;
                const nextUrl = this.currentSitemap.startUrls[this.currentPageIndex];
                
                // Add rate limiting delay
                const settings = await this.getSettings();
                setTimeout(() => {
                    this.navigateToUrl(nextUrl);
                }, settings.rateLimit || 1000);
            } else {
                // All pages processed
                await this.pauseScraping();
                this.broadcastMessage({ 
                    type: 'success', 
                    message: `Scraping completed! ${this.stats.records} records collected.` 
                });
            }
        }
    }

    async checkForNextPage(tabId) {
        try {
            const response = await chrome.tabs.sendMessage(tabId, {
                action: 'checkPagination'
            });
            
            if (response && response.nextUrl) {
                const settings = await this.getSettings();
                setTimeout(() => {
                    this.navigateToUrl(response.nextUrl);
                }, settings.rateLimit || 1000);
                return true;
            }
        } catch (error) {
            console.error('Error checking pagination:', error);
        }
        
        return false;
    }

    async getTabUrl(tabId) {
        try {
            const tab = await chrome.tabs.get(tabId);
            return tab.url;
        } catch (error) {
            return 'unknown';
        }
    }



    async getSitemaps() {
        const result = await chrome.storage.local.get(['sitemaps']);
        return result.sitemaps || [];
    }

    async saveSitemap(sitemap) {
        const sitemaps = await this.getSitemaps();
        const existingIndex = sitemaps.findIndex(s => s.id === sitemap.id);
        
        if (existingIndex >= 0) {
            sitemaps[existingIndex] = sitemap;
        } else {
            sitemap.id = Date.now().toString();
            sitemaps.push(sitemap);
        }

        await chrome.storage.local.set({ sitemaps });
    }

    async deleteSitemap(sitemapId) {
        const sitemaps = await this.getSitemaps();
        const filteredSitemaps = sitemaps.filter(s => s.id !== sitemapId);
        await chrome.storage.local.set({ sitemaps: filteredSitemaps });
    }

    async clearData() {
        this.scrapedData = [];
        this.stats = { records: 0, pages: 0, duplicates: 0 };
        await this.saveState();
        this.broadcastMessage({ type: 'statsUpdated', stats: this.stats });
    }

    async getSettings() {
        const result = await chrome.storage.local.get(['settings']);
        return result.settings || {
            rateLimit: 1000,
            maxPages: 100,
            infiniteScrollDelay: 2000,
            duplicateDetection: true
        };
    }

    async autoSave() {
        // Auto-save functionality - could export to downloads folder
        console.log('Auto-save triggered');
    }

    arrayToCSV(data, headers) {
        const csvRows = [headers.join(',')];
        
        for (const row of data) {
            const values = headers.map(header => {
                const value = row[header] || '';
                const stringValue = value.toString();
                // Escape quotes and wrap in quotes if contains comma, quote, or newline
                if (stringValue.includes(',') || stringValue.includes('"') || stringValue.includes('\n')) {
                    return `"${stringValue.replace(/"/g, '""')}"`;
                }
                return stringValue;
            });
            csvRows.push(values.join(','));
        }
        
        return csvRows.join('\n');
    }
    
    createSimpleXLSX(data, headers) {
        // Create a simple XML-based XLSX structure
        const escapeXML = (str) => {
            return str.toString()
                .replace(/&/g, '&amp;')
                .replace(/</g, '&lt;')
                .replace(/>/g, '&gt;')
                .replace(/"/g, '&quot;')
                .replace(/'/g, '&apos;');
        };
        
        let xml = '<?xml version="1.0" encoding="UTF-8"?>\n';
        xml += '<worksheet xmlns="http://schemas.openxmlformats.org/spreadsheetml/2006/main">\n';
        xml += '<sheetData>\n';
        
        // Header row
        xml += '<row r="1">\n';
        headers.forEach((header, index) => {
            const cellRef = String.fromCharCode(65 + index) + '1';
            xml += `<c r="${cellRef}" t="inlineStr"><is><t>${escapeXML(header)}</t></is></c>\n`;
        });
        xml += '</row>\n';
        
        // Data rows
        data.forEach((row, rowIndex) => {
            const rowNum = rowIndex + 2;
            xml += `<row r="${rowNum}">\n`;
            headers.forEach((header, colIndex) => {
                const cellRef = String.fromCharCode(65 + colIndex) + rowNum;
                const value = row[header] || '';
                xml += `<c r="${cellRef}" t="inlineStr"><is><t>${escapeXML(value)}</t></is></c>\n`;
            });
            xml += '</row>\n';
        });
        
        xml += '</sheetData>\n';
        xml += '</worksheet>';
        
        return xml;
    }

    broadcastMessage(message) {
        // Send message to all extension contexts (popup, options page)
        chrome.runtime.sendMessage(message).catch(() => {
            // Ignore errors if no listeners
        });
    }
}

// Initialize the scraping manager
const scrapingManager = new ScrapingManager();