class WebScraper {
    constructor() {
        this.isActive = false;
        this.currentSitemap = null;
        this.overlay = null;
        this.highlightedElement = null;
        this.scrapedData = [];
        this.mutationObserver = null;
        this.infiniteScrollHandler = null;
        this.recordHashes = new Set(); // For duplicate prevention
        this.isScrolling = false;
        this.scrollAttempts = 0;
        this.maxScrollAttempts = 50;
        this.lastScrollHeight = 0;
        this.noNewContentCount = 0;
        this.maxNoNewContentCount = 3;
        
        this.initializeListeners();
        this.createOverlay();
        this.setupMutationObserver();
    }

    initializeListeners() {
        // Listen for messages from background script
        chrome.runtime.onMessage.addListener((message, sender, sendResponse) => {
            this.handleMessage(message, sender, sendResponse);
            return true;
        });

        // Mouse events for element selection
        document.addEventListener('mouseover', (e) => this.handleMouseOver(e), true);
        document.addEventListener('mouseout', (e) => this.handleMouseOut(e), true);
        document.addEventListener('click', (e) => this.handleClick(e), true);
        
        // Keyboard events
        document.addEventListener('keydown', (e) => this.handleKeyDown(e), true);
    }

    createOverlay() {
        this.overlay = document.createElement('div');
        this.overlay.id = 'ai-spun-overlay';
        this.overlay.style.cssText = `
            position: absolute;
            pointer-events: none;
            border: 2px solid #007bff;
            background: rgba(0, 123, 255, 0.1);
            z-index: 999999;
            display: none;
            box-sizing: border-box;
        `;
        document.body.appendChild(this.overlay);
    }

    setupMutationObserver() {
        this.mutationObserver = new MutationObserver((mutations) => {
            if (!this.isActive || !this.currentSitemap) return;
            
            let hasNewContent = false;
            mutations.forEach((mutation) => {
                if (mutation.type === 'childList' && mutation.addedNodes.length > 0) {
                    // Check if new nodes contain elements matching our selectors
                    mutation.addedNodes.forEach((node) => {
                        if (node.nodeType === Node.ELEMENT_NODE) {
                            if (this.hasMatchingElements(node)) {
                                hasNewContent = true;
                            }
                        }
                    });
                }
            });
            
            if (hasNewContent && this.isScrolling) {
                this.noNewContentCount = 0;
                this.handleNewDynamicContent();
            }
        });
        
        this.mutationObserver.observe(document.body, {
            childList: true,
            subtree: true
        });
    }

    hasMatchingElements(node) {
        if (!this.currentSitemap || !this.currentSitemap.selectors) return false;
        
        for (const selector of this.currentSitemap.selectors) {
            if (selector.type === 'list' && node.matches && node.matches(selector.selector)) {
                return true;
            }
            if (node.querySelector && node.querySelector(selector.selector)) {
                return true;
            }
        }
        return false;
    }

    async handleNewDynamicContent() {
        // Extract data from newly loaded content
        const newData = await this.extractData(this.currentSitemap, true);
        if (newData.length > 0) {
            await chrome.runtime.sendMessage({
                action: 'dataScraped',
                data: newData
            });
        }
    }

    handleMessage(message, sender, sendResponse) {
        switch (message.action) {
            case 'startPageScraping':
                this.startPageScraping(message.sitemap);
                sendResponse({ success: true });
                break;
                
            case 'enableSelector':
                this.enableSelectorMode();
                sendResponse({ success: true });
                break;
                
            case 'disableSelector':
                this.disableSelectorMode();
                sendResponse({ success: true });
                break;
                
            case 'checkPagination':
                const nextUrl = this.checkPagination();
                sendResponse({ nextUrl });
                break;
                
            case 'testSelector':
                const elements = this.testSelector(message.selector);
                sendResponse({ elements: elements.length, preview: this.getElementPreview(elements) });
                break;
        }
    }

    async startPageScraping(sitemap) {
        this.currentSitemap = sitemap;
        this.scrapedData = [];
        
        try {
            // Wait for page to be fully loaded
            await this.waitForPageLoad();
            
            // Handle infinite scroll if configured
            if (this.hasInfiniteScrollSelector(sitemap)) {
                await this.handleInfiniteScroll(sitemap);
            }
            
            // Extract data using selectors
            const data = await this.extractData(sitemap);
            
            if (data.length > 0) {
                // Send data to background script
                await chrome.runtime.sendMessage({
                    action: 'dataScraped',
                    data: data
                });
            }
            
            // Notify page completion
            await chrome.runtime.sendMessage({
                action: 'pageComplete'
            });
            
        } catch (error) {
            console.error('Scraping error:', error);
            chrome.runtime.sendMessage({
                action: 'error',
                message: error.message
            });
        }
    }

    async waitForPageLoad() {
        return new Promise((resolve) => {
            if (document.readyState === 'complete') {
                resolve();
            } else {
                window.addEventListener('load', resolve, { once: true });
            }
        });
    }

    hasInfiniteScrollSelector(sitemap) {
        return sitemap.selectors.some(selector => selector.type === 'infiniteScroll');
    }

    async handleInfiniteScroll(sitemap) {
        this.isScrolling = true;
        this.scrollAttempts = 0;
        this.lastScrollHeight = document.body.scrollHeight;
        this.noNewContentCount = 0;
        
        const scrollDelay = sitemap.settings?.infiniteScrollDelay || 2000;
        const scrollStep = Math.max(window.innerHeight * 0.8, 500); // Scroll by viewport height
        const maxExecutionTime = 300000; // 5 minutes maximum execution time
        const startTime = Date.now();
        
        while (this.scrollAttempts < this.maxScrollAttempts && 
               this.noNewContentCount < this.maxNoNewContentCount &&
               (Date.now() - startTime) < maxExecutionTime) {
            const currentScrollTop = window.pageYOffset;
            const targetScroll = currentScrollTop + scrollStep;
            
            // Smooth scroll step by step
            await this.smoothScrollTo(targetScroll);
            
            // Wait for content to load
            await this.delay(scrollDelay);
            
            const currentHeight = document.body.scrollHeight;
            
            // Check if we've reached the bottom
            if (window.pageYOffset + window.innerHeight >= currentHeight - 100) {
                // Try scrolling to absolute bottom
                window.scrollTo(0, currentHeight);
                await this.delay(scrollDelay);
                
                const finalHeight = document.body.scrollHeight;
                if (finalHeight === currentHeight) {
                    this.noNewContentCount++;
                } else {
                    this.noNewContentCount = 0;
                    this.lastScrollHeight = finalHeight;
                }
            } else if (currentHeight === this.lastScrollHeight) {
                this.noNewContentCount++;
            } else {
                this.noNewContentCount = 0;
                this.lastScrollHeight = currentHeight;
            }
            
            this.scrollAttempts++;
            
            // Send progress update
            await chrome.runtime.sendMessage({
                action: 'scrollProgress',
                progress: {
                    attempts: this.scrollAttempts,
                    maxAttempts: this.maxScrollAttempts,
                    height: currentHeight,
                    noNewContentCount: this.noNewContentCount
                }
            });
        }
        
        this.isScrolling = false;
    }
    
    async smoothScrollTo(targetY) {
        return new Promise((resolve) => {
            const startY = window.pageYOffset;
            const distance = targetY - startY;
            const duration = 500; // 500ms scroll animation
            let startTime = null;
            
            function animation(currentTime) {
                if (startTime === null) startTime = currentTime;
                const timeElapsed = currentTime - startTime;
                const progress = Math.min(timeElapsed / duration, 1);
                
                // Easing function for smooth scroll
                const ease = progress * (2 - progress);
                window.scrollTo(0, startY + (distance * ease));
                
                if (progress < 1) {
                    requestAnimationFrame(animation);
                } else {
                    resolve();
                }
            }
            
            requestAnimationFrame(animation);
        });
    }

    async extractData(sitemap, onlyNew = false) {
        const results = [];
        
        for (const selector of sitemap.selectors) {
            if (selector.type === 'list') {
                const listData = this.extractListData(selector, onlyNew);
                results.push(...listData);
            } else if (selector.type !== 'pagination' && selector.type !== 'infiniteScroll') {
                const singleData = this.extractSingleData(selector);
                if (singleData !== null) {
                    const record = { [selector.id]: singleData };
                    if (!this.isDuplicate(record)) {
                        results.push(record);
                    }
                }
            }
        }
        
        return results;
    }
    
    generateRecordHash(record) {
        // Create a hash from the record's key fields to detect duplicates
        const keyFields = Object.keys(record).sort();
        const hashString = keyFields.map(key => `${key}:${record[key]}`).join('|');
        return btoa(hashString).replace(/[^a-zA-Z0-9]/g, ''); // Base64 encode and clean
    }
    
    isDuplicate(record) {
        const hash = this.generateRecordHash(record);
        if (this.recordHashes.has(hash)) {
            return true;
        }
        this.recordHashes.add(hash);
        return false;
    }

    extractListData(listSelector, onlyNew = false) {
        const elements = document.querySelectorAll(listSelector.selector);
        const results = [];
        
        elements.forEach((element, index) => {
            const item = {};
            
            if (listSelector.children) {
                listSelector.children.forEach(childSelector => {
                    const value = this.extractValueFromElement(element, childSelector);
                    if (value !== null) {
                        item[childSelector.id] = value;
                    }
                });
            }
            
            if (Object.keys(item).length > 0) {
                item._index = index;
                if (!this.isDuplicate(item)) {
                    results.push(item);
                }
            }
        });
        
        return results;
    }

    extractSingleData(selector) {
        if (selector.multiple) {
            const elements = document.querySelectorAll(selector.selector);
            const values = [];
            elements.forEach(element => {
                const value = this.extractValueFromElement(element, selector, true);
                if (value !== null) {
                    values.push(value);
                }
            });
            return values.length > 0 ? values : null;
        } else {
            return this.extractValueFromElement(document, selector);
        }
    }

    extractValueFromElement(context, selector, isDirectElement = false) {
        const element = isDirectElement ? context : 
            (context.querySelector ? 
                context.querySelector(selector.selector) : 
                context.querySelectorAll(selector.selector)[0]);
            
        if (!element) return null;
        
        switch (selector.type) {
            case 'text':
                return element.textContent?.trim() || '';
                
            case 'name':
                // Extract name with basic cleaning
                const nameText = element.textContent?.trim() || '';
                return nameText.replace(/\s+/g, ' ');
                
            case 'email':
                // Extract email with validation
                const emailText = element.textContent?.trim() || element.href?.replace('mailto:', '') || '';
                const emailMatch = emailText.match(/[a-zA-Z0-9._%+-]+@[a-zA-Z0-9.-]+\.[a-zA-Z]{2,}/);
                return emailMatch ? emailMatch[0] : emailText;
                
            case 'phone':
                // Extract phone number with basic formatting
                const phoneText = element.textContent?.trim() || element.href?.replace('tel:', '') || '';
                const phoneMatch = phoneText.match(/[\d\s\-\(\)\+\.]{7,}/);
                return phoneMatch ? phoneMatch[0].trim() : phoneText;
                
            case 'url':
                return element.href || element.src || '';
                
            case 'image':
                return element.src || element.dataset.src || '';
                
            case 'attribute':
                return element.getAttribute(selector.attribute) || '';
                
            case 'html':
                return element.innerHTML || '';
                
            case 'number':
                const text = element.textContent?.trim() || '';
                const number = parseFloat(text.replace(/[^0-9.-]/g, ''));
                return isNaN(number) ? null : number;
                
            case 'date':
                const dateText = element.textContent?.trim() || '';
                const date = new Date(dateText);
                return isNaN(date.getTime()) ? dateText : date.toISOString();
                
            default:
                return element.textContent?.trim() || '';
        }
    }

    checkPagination() {
        if (!this.currentSitemap) return null;
        
        const paginationSelector = this.currentSitemap.selectors.find(s => s.type === 'pagination');
        if (!paginationSelector) return null;
        
        const nextButton = document.querySelector(paginationSelector.selector);
        if (nextButton && !nextButton.disabled && nextButton.href) {
            return nextButton.href;
        }
        
        return null;
    }

    enableSelectorMode() {
        this.isActive = true;
        document.body.style.cursor = 'crosshair';
        this.showOverlay();
    }

    disableSelectorMode() {
        this.isActive = false;
        document.body.style.cursor = '';
        this.hideOverlay();
        this.clearHighlight();
    }

    handleMouseOver(event) {
        if (!this.isActive) return;
        
        event.preventDefault();
        event.stopPropagation();
        
        this.highlightElement(event.target);
    }

    handleMouseOut(event) {
        if (!this.isActive) return;
        
        event.preventDefault();
        event.stopPropagation();
    }

    handleClick(event) {
        if (!this.isActive) return;
        
        event.preventDefault();
        event.stopPropagation();
        
        const selector = this.generateSelector(event.target);
        
        // Send selector back to options page
        chrome.runtime.sendMessage({
            action: 'selectorSelected',
            selector: selector,
            element: {
                tagName: event.target.tagName,
                className: event.target.className,
                id: event.target.id,
                textContent: event.target.textContent?.substring(0, 100)
            }
        });
        
        this.disableSelectorMode();
    }

    handleKeyDown(event) {
        if (!this.isActive) return;
        
        if (event.key === 'Escape') {
            event.preventDefault();
            this.disableSelectorMode();
        }
    }

    highlightElement(element) {
        this.highlightedElement = element;
        const rect = element.getBoundingClientRect();
        
        this.overlay.style.display = 'block';
        this.overlay.style.left = (rect.left + window.scrollX) + 'px';
        this.overlay.style.top = (rect.top + window.scrollY) + 'px';
        this.overlay.style.width = rect.width + 'px';
        this.overlay.style.height = rect.height + 'px';
    }

    clearHighlight() {
        this.highlightedElement = null;
        this.overlay.style.display = 'none';
    }

    showOverlay() {
        this.overlay.style.display = 'block';
    }

    hideOverlay() {
        this.overlay.style.display = 'none';
    }

    generateSelector(element) {
        // Generate a unique CSS selector for the element
        const selectors = [];
        let current = element;
        
        while (current && current !== document.body) {
            let selector = current.tagName.toLowerCase();
            
            if (current.id) {
                selector += `#${current.id}`;
                selectors.unshift(selector);
                break;
            }
            
            if (current.className) {
                const classes = current.className.split(' ').filter(c => c.trim());
                if (classes.length > 0) {
                    selector += '.' + classes.join('.');
                }
            }
            
            // Add nth-child if needed for uniqueness
            const siblings = Array.from(current.parentNode?.children || []);
            const sameTagSiblings = siblings.filter(s => s.tagName === current.tagName);
            if (sameTagSiblings.length > 1) {
                const index = sameTagSiblings.indexOf(current) + 1;
                selector += `:nth-child(${index})`;
            }
            
            selectors.unshift(selector);
            current = current.parentNode;
        }
        
        return selectors.join(' > ');
    }

    testSelector(selectorString) {
        try {
            return Array.from(document.querySelectorAll(selectorString));
        } catch (error) {
            console.error('Invalid selector:', error);
            return [];
        }
    }

    getElementPreview(elements) {
        return elements.slice(0, 5).map(el => ({
            tagName: el.tagName,
            textContent: el.textContent?.substring(0, 50),
            className: el.className,
            id: el.id
        }));
    }

    delay(ms) {
        return new Promise(resolve => setTimeout(resolve, ms));
    }
}

// Initialize the web scraper
const webScraper = new WebScraper();