/**
 * AI API Integration utilities for AI Spun Webscraper
 * Handles Apify and OpenRouter API connections and data processing
 */

class AIAPIManager {
    constructor() {
        this.apifyConfig = null;
        this.openrouterConfig = null;
        this.loadConfigurations();
    }

    /**
     * Load API configurations from storage
     */
    async loadConfigurations() {
        try {
            const result = await chrome.storage.local.get(['apifyConfig', 'openrouterConfig']);
            this.apifyConfig = result.apifyConfig || null;
            this.openrouterConfig = result.openrouterConfig || null;
        } catch (error) {
            console.error('Failed to load API configurations:', error);
        }
    }

    /**
     * Save API configurations to storage
     */
    async saveConfigurations() {
        try {
            await chrome.storage.local.set({
                apifyConfig: this.apifyConfig,
                openrouterConfig: this.openrouterConfig
            });
        } catch (error) {
            console.error('Failed to save API configurations:', error);
            throw error;
        }
    }

    /**
     * Configure Apify API
     */
    async configureApify(apiKey, baseUrl = 'https://api.apify.com/v2') {
        this.apifyConfig = {
            apiKey: apiKey,
            baseUrl: baseUrl,
            enabled: true
        };
        await this.saveConfigurations();
    }

    /**
     * Configure OpenRouter API
     */
    async configureOpenRouter(apiKey, baseUrl = 'https://openrouter.ai/api/v1', model = 'anthropic/claude-3-sonnet') {
        this.openrouterConfig = {
            apiKey: apiKey,
            baseUrl: baseUrl,
            model: model,
            enabled: true
        };
        await this.saveConfigurations();
    }

    /**
     * Test Apify API connection
     */
    async testApifyConnection() {
        if (!this.apifyConfig || !this.apifyConfig.apiKey) {
            throw new Error('Apify API key not configured');
        }

        try {
            const response = await fetch(`${this.apifyConfig.baseUrl}/users/me`, {
                headers: {
                    'Authorization': `Bearer ${this.apifyConfig.apiKey}`,
                    'Content-Type': 'application/json'
                }
            });

            if (!response.ok) {
                throw new Error(`HTTP ${response.status}: ${response.statusText}`);
            }

            const data = await response.json();
            return {
                success: true,
                data: data,
                message: 'Connection successful'
            };
        } catch (error) {
            return {
                success: false,
                error: error.message,
                message: 'Connection failed'
            };
        }
    }

    /**
     * Test OpenRouter API connection
     */
    async testOpenRouterConnection() {
        if (!this.openrouterConfig || !this.openrouterConfig.apiKey) {
            throw new Error('OpenRouter API key not configured');
        }

        try {
            const response = await fetch(`${this.openrouterConfig.baseUrl}/models`, {
                headers: {
                    'Authorization': `Bearer ${this.openrouterConfig.apiKey}`,
                    'Content-Type': 'application/json'
                }
            });

            if (!response.ok) {
                throw new Error(`HTTP ${response.status}: ${response.statusText}`);
            }

            const data = await response.json();
            return {
                success: true,
                data: data,
                message: 'Connection successful'
            };
        } catch (error) {
            return {
                success: false,
                error: error.message,
                message: 'Connection failed'
            };
        }
    }

    /**
     * Process data with OpenRouter AI
     */
    async processWithAI(data, task = 'clean', options = {}) {
        if (!this.openrouterConfig || !this.openrouterConfig.enabled) {
            throw new Error('OpenRouter not configured or disabled');
        }

        const prompts = {
            clean: `Clean and standardize the following data. Remove duplicates, fix formatting, and ensure consistency. Return only the cleaned data in JSON format:\n\n${JSON.stringify(data)}`,
            summarize: `Analyze and summarize the following scraped data. Provide key insights and patterns. Return a structured summary:\n\n${JSON.stringify(data)}`,
            classify: `Classify and categorize the following data. Add relevant tags and categories. Return the data with added classification:\n\n${JSON.stringify(data)}`,
            extract: `Extract key information from the following data. Focus on names, emails, phones, and important details:\n\n${JSON.stringify(data)}`
        };

        try {
            const response = await fetch(`${this.openrouterConfig.baseUrl}/chat/completions`, {
                method: 'POST',
                headers: {
                    'Authorization': `Bearer ${this.openrouterConfig.apiKey}`,
                    'Content-Type': 'application/json',
                    'HTTP-Referer': 'https://spunwebtechnology.com',
                    'X-Title': 'AI Spun Webscraper'
                },
                body: JSON.stringify({
                    model: this.openrouterConfig.model,
                    messages: [
                        {
                            role: 'system',
                            content: 'You are a data processing assistant for a web scraping tool. Always return valid JSON responses.'
                        },
                        {
                            role: 'user',
                            content: prompts[task] || prompts.clean
                        }
                    ],
                    max_tokens: options.maxTokens || 2000,
                    temperature: options.temperature || 0.1
                })
            });

            if (!response.ok) {
                throw new Error(`OpenRouter API error: ${response.status}`);
            }

            const result = await response.json();
            const content = result.choices[0]?.message?.content;

            try {
                return JSON.parse(content);
            } catch (parseError) {
                return {
                    processed: true,
                    original: data,
                    aiResponse: content,
                    note: 'AI response was not valid JSON'
                };
            }
        } catch (error) {
            console.error('AI processing error:', error);
            throw error;
        }
    }

    /**
     * Enhanced scraping with Apify
     */
    async scrapeWithApify(url, options = {}) {
        if (!this.apifyConfig || !this.apifyConfig.enabled) {
            throw new Error('Apify not configured or disabled');
        }

        const actorId = options.actorId || 'apify/web-scraper';
        const input = {
            startUrls: [{ url: url }],
            linkSelector: options.linkSelector || 'a[href]',
            pageFunction: options.pageFunction || `
                async function pageFunction(context) {
                    const { page, request } = context;
                    const title = await page.title();
                    const content = await page.evaluate(() => document.body.innerText);
                    return {
                        url: request.url,
                        title: title,
                        content: content.substring(0, 1000)
                    };
                }
            `,
            maxRequestsPerCrawl: options.maxPages || 10,
            ...options.apifyOptions
        };

        try {
            // Start the actor run
            const runResponse = await fetch(`${this.apifyConfig.baseUrl}/acts/${actorId}/runs`, {
                method: 'POST',
                headers: {
                    'Authorization': `Bearer ${this.apifyConfig.apiKey}`,
                    'Content-Type': 'application/json'
                },
                body: JSON.stringify(input)
            });

            if (!runResponse.ok) {
                throw new Error(`Failed to start Apify actor: ${runResponse.status}`);
            }

            const runData = await runResponse.json();
            const runId = runData.data.id;

            // Poll for completion
            let attempts = 0;
            const maxAttempts = 60; // 5 minutes max
            
            while (attempts < maxAttempts) {
                await new Promise(resolve => setTimeout(resolve, 5000)); // Wait 5 seconds
                
                const statusResponse = await fetch(`${this.apifyConfig.baseUrl}/acts/${actorId}/runs/${runId}`, {
                    headers: {
                        'Authorization': `Bearer ${this.apifyConfig.apiKey}`
                    }
                });

                const statusData = await statusResponse.json();
                const status = statusData.data.status;

                if (status === 'SUCCEEDED') {
                    // Get the results
                    const resultsResponse = await fetch(`${this.apifyConfig.baseUrl}/acts/${actorId}/runs/${runId}/dataset/items`, {
                        headers: {
                            'Authorization': `Bearer ${this.apifyConfig.apiKey}`
                        }
                    });

                    const results = await resultsResponse.json();
                    return {
                        success: true,
                        data: results,
                        runId: runId
                    };
                } else if (status === 'FAILED' || status === 'ABORTED') {
                    throw new Error(`Apify run ${status.toLowerCase()}`);
                }

                attempts++;
            }

            throw new Error('Apify run timed out');
        } catch (error) {
            console.error('Apify scraping error:', error);
            throw error;
        }
    }

    /**
     * Get API status
     */
    getStatus() {
        return {
            apify: {
                configured: !!(this.apifyConfig && this.apifyConfig.apiKey),
                enabled: !!(this.apifyConfig && this.apifyConfig.enabled)
            },
            openrouter: {
                configured: !!(this.openrouterConfig && this.openrouterConfig.apiKey),
                enabled: !!(this.openrouterConfig && this.openrouterConfig.enabled)
            }
        };
    }

    /**
     * Disable API
     */
    async disableAPI(apiName) {
        if (apiName === 'apify' && this.apifyConfig) {
            this.apifyConfig.enabled = false;
        } else if (apiName === 'openrouter' && this.openrouterConfig) {
            this.openrouterConfig.enabled = false;
        }
        await this.saveConfigurations();
    }

    /**
     * Enable API
     */
    async enableAPI(apiName) {
        if (apiName === 'apify' && this.apifyConfig) {
            this.apifyConfig.enabled = true;
        } else if (apiName === 'openrouter' && this.openrouterConfig) {
            this.openrouterConfig.enabled = true;
        }
        await this.saveConfigurations();
    }
}

// Export for use in other modules
if (typeof module !== 'undefined' && module.exports) {
    module.exports = AIAPIManager;
} else if (typeof window !== 'undefined') {
    window.AIAPIManager = AIAPIManager;
}