/**
 * Export utilities for AI Spun Webscraper
 * Lightweight CSV and basic XLSX export functionality
 */

class ExportUtils {
    /**
     * Convert data array to CSV format
     * @param {Array} data - Array of objects to export
     * @param {Array} headers - Optional array of headers
     * @returns {string} CSV formatted string
     */
    static arrayToCSV(data, headers = null) {
        if (!data || data.length === 0) {
            return '';
        }

        // Get headers from first object if not provided
        if (!headers) {
            headers = Object.keys(data[0]);
        }

        // Escape CSV values
        const escapeCSV = (value) => {
            if (value === null || value === undefined) {
                return '';
            }
            
            const stringValue = String(value);
            
            // If value contains comma, newline, or quote, wrap in quotes and escape quotes
            if (stringValue.includes(',') || stringValue.includes('\n') || stringValue.includes('"')) {
                return '"' + stringValue.replace(/"/g, '""') + '"';
            }
            
            return stringValue;
        };

        // Create CSV content
        const csvHeaders = headers.map(escapeCSV).join(',');
        const csvRows = data.map(row => 
            headers.map(header => escapeCSV(row[header])).join(',')
        );

        return [csvHeaders, ...csvRows].join('\n');
    }

    /**
     * Generate timestamped filename
     * @param {string} baseName - Base filename
     * @param {string} extension - File extension
     * @returns {string} Timestamped filename
     */
    static generateTimestampedFilename(baseName, extension) {
        const now = new Date();
        const timestamp = now.toISOString().replace(/[:.]/g, '-').slice(0, -5);
        return `${baseName}_${timestamp}.${extension}`;
    }

    /**
     * Download CSV file
     * @param {Array} data - Data to export
     * @param {string} filename - Filename without extension
     * @param {Array} headers - Optional headers
     */
    static downloadCSV(data, filename = 'scraped_data', headers = null) {
        const timestampedFilename = this.generateTimestampedFilename(filename, 'csv');
        const csvContent = this.arrayToCSV(data, headers);
        const blob = new Blob([csvContent], { type: 'text/csv;charset=utf-8;' });
        
        // Use Chrome downloads API if available (for extensions)
        if (typeof chrome !== 'undefined' && chrome.downloads) {
            const url = URL.createObjectURL(blob);
            chrome.downloads.download({
                url: url,
                filename: timestampedFilename,
                saveAs: false
            }, (downloadId) => {
                if (chrome.runtime.lastError) {
                    console.error('Download failed:', chrome.runtime.lastError);
                    // Fallback to regular download
                    this.fallbackDownload(url, timestampedFilename);
                }
                URL.revokeObjectURL(url);
            });
        } else {
            // Fallback for regular web pages
            const url = URL.createObjectURL(blob);
            this.fallbackDownload(url, timestampedFilename);
            URL.revokeObjectURL(url);
        }
    }
    
    static fallbackDownload(url, filename) {
        const a = document.createElement('a');
        a.href = url;
        a.download = filename;
        document.body.appendChild(a);
        a.click();
        document.body.removeChild(a);
    }

    /**
     * Create XLSX file using SheetJS (if available) or fallback to simplified format
     * @param {Array} data - Array of objects to export
     * @param {Array} headers - Optional array of headers
     * @returns {Blob} XLSX file blob
     */
    static arrayToXLSX(data, headers = null) {
        if (!data || data.length === 0) {
            return new Blob([''], { type: 'application/vnd.openxmlformats-officedocument.spreadsheetml.sheet' });
        }

        // Check if SheetJS (XLSX) is available
        if (typeof XLSX !== 'undefined') {
            return this.createXLSXWithSheetJS(data, headers);
        }

        // Fallback to simplified XML format
        return this.createSimpleXLSXBlob(data, headers);
    }

    /**
     * Create XLSX using SheetJS library
     * @param {Array} data - Array of objects to export
     * @param {Array} headers - Optional array of headers
     * @returns {Blob} XLSX file blob
     */
    static createXLSXWithSheetJS(data, headers = null) {
        // Get headers from first object if not provided
        if (!headers) {
            headers = Object.keys(data[0]);
        }

        // Create worksheet data
        const wsData = [headers];
        data.forEach(row => {
            wsData.push(headers.map(header => row[header] || ''));
        });

        // Create workbook and worksheet
        const wb = XLSX.utils.book_new();
        const ws = XLSX.utils.aoa_to_sheet(wsData);
        
        // Add worksheet to workbook
        XLSX.utils.book_append_sheet(wb, ws, 'Sheet1');
        
        // Generate XLSX file
        const xlsxData = XLSX.write(wb, { bookType: 'xlsx', type: 'array' });
        
        return new Blob([xlsxData], { 
            type: 'application/vnd.openxmlformats-officedocument.spreadsheetml.sheet' 
        });
    }

    /**
     * Create a basic XLSX file (simplified XML format) - fallback method
     * @param {Array} data - Array of objects to export
     * @param {Array} headers - Optional array of headers
     * @returns {Blob} XLSX file blob
     */
    static createSimpleXLSXBlob(data, headers = null) {
        // Get headers from first object if not provided
        if (!headers) {
            headers = Object.keys(data[0]);
        }

        // Escape XML characters
        const escapeXML = (value) => {
            if (value === null || value === undefined) {
                return '';
            }
            
            return String(value)
                .replace(/&/g, '&amp;')
                .replace(/</g, '&lt;')
                .replace(/>/g, '&gt;')
                .replace(/"/g, '&quot;')
                .replace(/'/g, '&#39;');
        };

        // Create worksheet XML
        let worksheetXML = `<?xml version="1.0" encoding="UTF-8" standalone="yes"?>
<worksheet xmlns="http://schemas.openxmlformats.org/spreadsheetml/2006/main">
<sheetData>`;

        // Add header row
        worksheetXML += '<row r="1">';
        headers.forEach((header, index) => {
            const cellRef = this.numberToColumnLetter(index + 1) + '1';
            worksheetXML += `<c r="${cellRef}" t="inlineStr"><is><t>${escapeXML(header)}</t></is></c>`;
        });
        worksheetXML += '</row>';

        // Add data rows
        data.forEach((row, rowIndex) => {
            const rowNum = rowIndex + 2;
            worksheetXML += `<row r="${rowNum}">`;
            
            headers.forEach((header, colIndex) => {
                const cellRef = this.numberToColumnLetter(colIndex + 1) + rowNum;
                const value = row[header];
                
                if (typeof value === 'number') {
                    worksheetXML += `<c r="${cellRef}"><v>${value}</v></c>`;
                } else {
                    worksheetXML += `<c r="${cellRef}" t="inlineStr"><is><t>${escapeXML(value)}</t></is></c>`;
                }
            });
            
            worksheetXML += '</row>';
        });

        worksheetXML += '</sheetData></worksheet>';

        // Create a simple XLSX structure (this is a simplified version)
        // In a real implementation, you'd need proper ZIP compression and more XML files
        const xlsxContent = this.createSimpleXLSX(worksheetXML);
        
        return new Blob([xlsxContent], { 
            type: 'application/vnd.openxmlformats-officedocument.spreadsheetml.sheet' 
        });
    }

    /**
     * Download XLSX file
     * @param {Array} data - Data to export
     * @param {string} filename - Filename without extension
     * @param {Array} headers - Optional headers
     */
    static downloadXLSX(data, filename = 'scraped_data', headers = null) {
        const timestampedFilename = this.generateTimestampedFilename(filename, 'xlsx');
        const blob = this.arrayToXLSX(data, headers);
        
        // Use Chrome downloads API if available (for extensions)
        if (typeof chrome !== 'undefined' && chrome.downloads) {
            const url = URL.createObjectURL(blob);
            chrome.downloads.download({
                url: url,
                filename: timestampedFilename,
                saveAs: false
            }, (downloadId) => {
                if (chrome.runtime.lastError) {
                    console.error('Download failed:', chrome.runtime.lastError);
                    // Fallback to regular download
                    this.fallbackDownload(url, timestampedFilename);
                }
                URL.revokeObjectURL(url);
            });
        } else {
            // Fallback for regular web pages
            const url = URL.createObjectURL(blob);
            this.fallbackDownload(url, timestampedFilename);
            URL.revokeObjectURL(url);
        }
    }

    /**
     * Convert number to Excel column letter (1 = A, 2 = B, etc.)
     * @param {number} num - Column number
     * @returns {string} Column letter
     */
    static numberToColumnLetter(num) {
        let result = '';
        while (num > 0) {
            num--;
            result = String.fromCharCode(65 + (num % 26)) + result;
            num = Math.floor(num / 26);
        }
        return result;
    }

    /**
     * Create a simplified XLSX file structure
     * @param {string} worksheetXML - Worksheet XML content
     * @returns {string} XLSX content as string
     */
    static createSimpleXLSX(worksheetXML) {
        // This is a very simplified XLSX format
        // For production use, consider using a proper library like SheetJS
        
        // Create basic XML structure that Excel can read
        const contentTypes = `<?xml version="1.0" encoding="UTF-8" standalone="yes"?>
<Types xmlns="http://schemas.openxmlformats.org/package/2006/content-types">
<Default Extension="rels" ContentType="application/vnd.openxmlformats-package.relationships+xml"/>
<Default Extension="xml" ContentType="application/xml"/>
<Override PartName="/xl/workbook.xml" ContentType="application/vnd.openxmlformats-officedocument.spreadsheetml.sheet.main+xml"/>
<Override PartName="/xl/worksheets/sheet1.xml" ContentType="application/vnd.openxmlformats-officedocument.spreadsheetml.worksheet+xml"/>
</Types>`;

        const workbook = `<?xml version="1.0" encoding="UTF-8" standalone="yes"?>
<workbook xmlns="http://schemas.openxmlformats.org/spreadsheetml/2006/main">
<sheets>
<sheet name="Sheet1" sheetId="1" r:id="rId1" xmlns:r="http://schemas.openxmlformats.org/officeDocument/2006/relationships"/>
</sheets>
</workbook>`;

        // For simplicity, we'll return the worksheet XML as a basic Excel-compatible format
        // Note: This won't create a proper XLSX file, but will create an XML file that Excel can open
        return worksheetXML;
    }

    /**
     * Download a blob as a file
     * @param {Blob} blob - Blob to download
     * @param {string} filename - Filename
     */
    static downloadBlob(blob, filename) {
        const url = URL.createObjectURL(blob);
        const link = document.createElement('a');
        link.href = url;
        link.download = filename;
        link.style.display = 'none';
        
        document.body.appendChild(link);
        link.click();
        document.body.removeChild(link);
        
        // Clean up the URL object
        setTimeout(() => URL.revokeObjectURL(url), 100);
    }

    /**
     * Generate filename with timestamp
     * @param {string} sitemapName - Name of the sitemap
     * @param {string} extension - File extension (csv or xlsx)
     * @returns {string} Generated filename
     */
    static generateFilename(sitemapName = 'data', extension = 'csv') {
        const timestamp = new Date().toISOString().replace(/[:.]/g, '-').slice(0, -5);
        const safeName = sitemapName.replace(/[^a-zA-Z0-9]/g, '-').toLowerCase();
        return `ai-spun-${safeName}-${timestamp}.${extension}`;
    }

    /**
     * Filter data by date range
     * @param {Array} data - Data array
     * @param {string} dateField - Field name containing date
     * @param {Date} startDate - Start date
     * @param {Date} endDate - End date
     * @returns {Array} Filtered data
     */
    static filterByDateRange(data, dateField = '_scrapedAt', startDate = null, endDate = null) {
        if (!startDate && !endDate) {
            return data;
        }

        return data.filter(item => {
            const itemDate = new Date(item[dateField]);
            
            if (startDate && itemDate < startDate) {
                return false;
            }
            
            if (endDate && itemDate > endDate) {
                return false;
            }
            
            return true;
        });
    }

    /**
     * Filter data by sitemap
     * @param {Array} data - Data array
     * @param {string} sitemapId - Sitemap ID to filter by
     * @returns {Array} Filtered data
     */
    static filterBySitemap(data, sitemapId) {
        if (!sitemapId) {
            return data;
        }

        return data.filter(item => item._sitemapId === sitemapId);
    }

    /**
     * Get export statistics
     * @param {Array} data - Data to analyze
     * @returns {Object} Export statistics
     */
    static getExportStats(data) {
        if (!data || data.length === 0) {
            return {
                totalRecords: 0,
                sitemaps: [],
                dateRange: null,
                fields: []
            };
        }

        const sitemaps = [...new Set(data.map(item => item._sitemapId).filter(Boolean))];
        const dates = data.map(item => new Date(item._scrapedAt)).filter(date => !isNaN(date));
        const fields = [...new Set(data.flatMap(item => Object.keys(item)))];

        return {
            totalRecords: data.length,
            sitemaps: sitemaps,
            dateRange: dates.length > 0 ? {
                earliest: new Date(Math.min(...dates)),
                latest: new Date(Math.max(...dates))
            } : null,
            fields: fields
        };
    }
}

// Export for use in other modules
if (typeof module !== 'undefined' && module.exports) {
    module.exports = ExportUtils;
} else if (typeof window !== 'undefined') {
    window.ExportUtils = ExportUtils;
}