class OptionsController {
    constructor() {
        this.currentSitemap = null;
        this.currentSitemapId = null;
        this.sitemaps = [];
        this.scrapedData = [];
        this.settings = {};
        
        this.initializeElements();
        this.bindEvents();
        this.loadData();
        this.initializeTabs();
    }

    initializeElements() {
        // Tab elements
        this.tabButtons = document.querySelectorAll('.tab-button');
        this.tabContents = document.querySelectorAll('.tab-content');
        
        // Sitemap elements
        this.sitemapsList = document.getElementById('sitemapsList');
        this.newSitemapBtn = document.getElementById('newSitemapBtn');
        this.createFirstSitemapBtn = document.getElementById('createFirstSitemapBtn');
        this.sitemapModal = document.getElementById('sitemapModal');
        this.sitemapForm = document.getElementById('sitemapForm');
        
        // Selector elements
        this.selectorsGrid = document.getElementById('selectorsGrid');
        this.newSelectorBtn = document.getElementById('newSelectorBtn');
        this.selectorModal = document.getElementById('selectorModal');
        this.selectorForm = document.getElementById('selectorForm');
        this.currentSitemapInfo = document.getElementById('currentSitemapInfo');
        
        // Settings elements
        this.settingsForm = {
            rateLimit: document.getElementById('rateLimit'),
            maxPages: document.getElementById('maxPages'),
            infiniteScrollDelay: document.getElementById('infiniteScrollDelay'),
            duplicateDetection: document.getElementById('duplicateDetection')
        };
        
        // Data elements
        this.dataTable = document.getElementById('dataTable');
        this.dataTableHead = document.getElementById('dataTableHead');
        this.dataTableBody = document.getElementById('dataTableBody');
        this.dataStats = document.getElementById('dataStats');
        
        // Export elements
        this.exportForm = {
            format: document.querySelectorAll('input[name="exportFormat"]'),
            sitemap: document.getElementById('exportSitemap'),
            dateFrom: document.getElementById('exportDateFrom'),
            dateTo: document.getElementById('exportDateTo')
        };
        
        // AI APIs elements
        this.aiApiElements = {
            apifyApiKey: document.getElementById('apifyApiKey'),
            openrouterApiKey: document.getElementById('openrouterApiKey'),
            aiProcessingEnabled: document.getElementById('aiProcessingEnabled'),
            aiModel: document.getElementById('aiModel'),
            aiPrompt: document.getElementById('aiPrompt'),
            apifyStatus: document.getElementById('apifyStatus'),
            openrouterStatus: document.getElementById('openrouterStatus')
        };
    }

    bindEvents() {
        // Tab navigation
        this.tabButtons.forEach(button => {
            button.addEventListener('click', (e) => {
                this.switchTab(e.target.dataset.tab);
            });
        });

        // Sitemap events
        this.newSitemapBtn?.addEventListener('click', () => this.openSitemapModal());
        this.createFirstSitemapBtn?.addEventListener('click', () => this.openSitemapModal());
        document.getElementById('closeSitemapModal')?.addEventListener('click', () => this.closeSitemapModal());
        document.getElementById('saveSitemapBtn')?.addEventListener('click', () => this.saveSitemap());
        document.getElementById('cancelSitemapBtn')?.addEventListener('click', () => this.closeSitemapModal());
        document.getElementById('addUrlBtn')?.addEventListener('click', () => this.addUrlInput());

        // Selector events
        this.newSelectorBtn?.addEventListener('click', () => this.openSelectorModal());
        document.getElementById('closeSelectorModal')?.addEventListener('click', () => this.closeSelectorModal());
        document.getElementById('saveSelectorBtn')?.addEventListener('click', () => this.saveSelector());
        document.getElementById('cancelSelectorBtn')?.addEventListener('click', () => this.closeSelectorModal());
        document.getElementById('pickSelectorBtn')?.addEventListener('click', () => this.pickSelector());
        document.getElementById('testSelectorBtn')?.addEventListener('click', () => this.testSelector());
        document.getElementById('selectSitemapBtn')?.addEventListener('click', () => this.selectSitemapForEditing());
        
        // Selector type change
        document.getElementById('selectorType')?.addEventListener('change', (e) => {
            this.handleSelectorTypeChange(e.target.value);
        });

        // Settings events
        document.getElementById('saveSettingsBtn')?.addEventListener('click', () => this.saveSettings());
        document.getElementById('resetSettingsBtn')?.addEventListener('click', () => this.resetSettings());

        // Data events
        document.getElementById('refreshDataBtn')?.addEventListener('click', () => this.refreshData());
        document.getElementById('clearDataBtn')?.addEventListener('click', () => this.clearData());
        document.getElementById('searchData')?.addEventListener('input', (e) => this.filterData());
        document.getElementById('filterSitemap')?.addEventListener('change', (e) => this.filterData());

        // Export events
        document.getElementById('exportDataBtn')?.addEventListener('click', () => this.exportData());
        this.exportForm.format?.forEach(radio => {
            radio.addEventListener('change', () => this.updateExportPreview());
        });
        
        // AI APIs events
        document.getElementById('saveAiConfigBtn')?.addEventListener('click', () => this.saveAiConfig());
        document.getElementById('testApifyBtn')?.addEventListener('click', () => this.testApifyConnection());
        document.getElementById('testOpenrouterBtn')?.addEventListener('click', () => this.testOpenrouterConnection());
        
        this.exportForm.sitemap?.addEventListener('change', () => this.updateExportPreview());
        this.exportForm.dateFrom?.addEventListener('change', () => this.updateExportPreview());
        this.exportForm.dateTo?.addEventListener('change', () => this.updateExportPreview());

        // Listen for messages from content script
        chrome.runtime.onMessage.addListener((message, sender, sendResponse) => {
            this.handleMessage(message);
        });

        // Modal close on backdrop click
        document.addEventListener('click', (e) => {
            if (e.target.classList.contains('modal')) {
                this.closeAllModals();
            }
        });
    }

    initializeTabs() {
        // Set initial tab
        this.switchTab('sitemaps');
    }

    switchTab(tabName) {
        // Update tab buttons
        this.tabButtons.forEach(button => {
            button.classList.toggle('active', button.dataset.tab === tabName);
        });

        // Update tab contents
        this.tabContents.forEach(content => {
            content.classList.toggle('active', content.id === tabName);
        });

        // Load tab-specific data
        switch (tabName) {
            case 'sitemaps':
                this.loadSitemaps();
                break;
            case 'selectors':
                this.loadSelectors();
                break;
            case 'settings':
                this.loadSettings();
                break;
            case 'data':
                this.loadData();
                break;
            case 'export':
                this.loadExportOptions();
                break;
            case 'ai-apis':
                this.loadAiConfig();
                break;
            case 'about':
                this.loadAbout();
                break;
        }
    }

    async loadData() {
        try {
            const result = await chrome.storage.local.get([
                'sitemaps',
                'scrapedData',
                'settings',
                'currentSitemap'
            ]);

            this.sitemaps = result.sitemaps || [];
            this.scrapedData = result.scrapedData || [];
            this.settings = result.settings || this.getDefaultSettings();
            this.currentSitemap = result.currentSitemap || null;
        } catch (error) {
            console.error('Error loading data:', error);
        }
    }

    getDefaultSettings() {
        return {
            rateLimit: 1000,
            maxPages: 100,
            infiniteScrollDelay: 2000,
            duplicateDetection: true
        };
    }

    // Sitemap Management
    async loadSitemaps() {
        await this.loadData();
        this.renderSitemaps();
    }

    renderSitemaps() {
        if (this.sitemaps.length === 0) {
            this.sitemapsList.innerHTML = `
                <div class="empty-state">
                    <div class="empty-icon">🗺️</div>
                    <h3>No sitemaps yet</h3>
                    <p>Create your first sitemap to start scraping websites</p>
                    <button class="btn btn-primary" id="createFirstSitemapBtn">Create Sitemap</button>
                </div>
            `;
            
            // Re-bind the event
            document.getElementById('createFirstSitemapBtn')?.addEventListener('click', () => this.openSitemapModal());
            return;
        }

        const sitemapsHtml = this.sitemaps.map(sitemap => `
            <div class="sitemap-card" data-sitemap-id="${sitemap.id}">
                <div class="sitemap-header">
                    <div>
                        <div class="sitemap-name">${this.escapeHtml(sitemap.name)}</div>
                        <div class="sitemap-meta">
                            Created: ${new Date(sitemap.createdAt || Date.now()).toLocaleDateString()}
                            • ${sitemap.selectors?.length || 0} selectors
                        </div>
                    </div>
                    <div class="sitemap-actions">
                        <button class="btn btn-small btn-secondary" onclick="optionsController.editSitemap('${sitemap.id}')">Edit</button>
                        <button class="btn btn-small btn-primary" onclick="optionsController.selectSitemap('${sitemap.id}')">Select</button>
                        <button class="btn btn-small btn-danger" onclick="optionsController.deleteSitemap('${sitemap.id}')">Delete</button>
                    </div>
                </div>
                
                <div class="sitemap-urls">
                    <h4>Start URLs:</h4>
                    <div class="url-list-display">
                        ${(sitemap.startUrls || []).map(url => `<span class="url-tag">${this.escapeHtml(url)}</span>`).join('')}
                    </div>
                </div>
                
                <div class="sitemap-selectors">
                    ${(sitemap.selectors || []).map(selector => `<span class="selector-tag">${this.escapeHtml(selector.id)} (${selector.type})</span>`).join('')}
                </div>
            </div>
        `).join('');

        this.sitemapsList.innerHTML = sitemapsHtml;
    }

    openSitemapModal(sitemapId = null) {
        const modal = this.sitemapModal;
        const title = document.getElementById('sitemapModalTitle');
        const form = this.sitemapForm;

        if (sitemapId) {
            const sitemap = this.sitemaps.find(s => s.id === sitemapId);
            if (sitemap) {
                title.textContent = 'Edit Sitemap';
                document.getElementById('sitemapName').value = sitemap.name;
                this.populateUrlInputs(sitemap.startUrls || []);
                this.currentSitemapId = sitemapId;
            }
        } else {
            title.textContent = 'New Sitemap';
            form.reset();
            this.populateUrlInputs(['']);
            this.currentSitemapId = null;
        }

        modal.classList.add('active');
    }

    closeSitemapModal() {
        this.sitemapModal.classList.remove('active');
        this.currentSitemapId = null;
    }

    populateUrlInputs(urls) {
        const urlList = document.getElementById('urlList');
        urlList.innerHTML = '';

        urls.forEach((url, index) => {
            this.addUrlInput(url);
        });

        if (urls.length === 0) {
            this.addUrlInput();
        }
    }

    addUrlInput(value = '') {
        const urlList = document.getElementById('urlList');
        const urlGroup = document.createElement('div');
        urlGroup.className = 'url-input-group';
        urlGroup.innerHTML = `
            <input type="url" class="url-input" placeholder="https://example.com" value="${this.escapeHtml(value)}" required>
            <button type="button" class="btn btn-small btn-danger remove-url">Remove</button>
        `;

        urlGroup.querySelector('.remove-url').addEventListener('click', () => {
            if (urlList.children.length > 1) {
                urlGroup.remove();
            }
        });

        urlList.appendChild(urlGroup);
    }

    async saveSitemap() {
        const form = this.sitemapForm;
        const formData = new FormData(form);
        
        const name = document.getElementById('sitemapName').value.trim();
        if (!name) {
            alert('Please enter a sitemap name');
            return;
        }

        const urlInputs = document.querySelectorAll('.url-input');
        const startUrls = Array.from(urlInputs)
            .map(input => input.value.trim())
            .filter(url => url);

        if (startUrls.length === 0) {
            alert('Please enter at least one start URL');
            return;
        }

        const sitemap = {
            id: this.currentSitemapId || Date.now().toString(),
            name: name,
            startUrls: startUrls,
            selectors: this.currentSitemapId ? 
                (this.sitemaps.find(s => s.id === this.currentSitemapId)?.selectors || []) : [],
            createdAt: this.currentSitemapId ? 
                (this.sitemaps.find(s => s.id === this.currentSitemapId)?.createdAt || Date.now()) : Date.now(),
            updatedAt: Date.now()
        };

        try {
            await chrome.runtime.sendMessage({
                action: 'saveSitemap',
                sitemap: sitemap
            });

            this.closeSitemapModal();
            this.loadSitemaps();
        } catch (error) {
            console.error('Error saving sitemap:', error);
            alert('Failed to save sitemap');
        }
    }

    async editSitemap(sitemapId) {
        this.openSitemapModal(sitemapId);
    }

    async selectSitemap(sitemapId) {
        const sitemap = this.sitemaps.find(s => s.id === sitemapId);
        if (sitemap) {
            await chrome.storage.local.set({ currentSitemap: sitemap });
            this.currentSitemap = sitemap;
            alert(`Selected sitemap: ${sitemap.name}`);
            this.updateCurrentSitemapDisplay();
        }
    }

    async deleteSitemap(sitemapId) {
        if (confirm('Are you sure you want to delete this sitemap?')) {
            try {
                await chrome.runtime.sendMessage({
                    action: 'deleteSitemap',
                    sitemapId: sitemapId
                });
                this.loadSitemaps();
            } catch (error) {
                console.error('Error deleting sitemap:', error);
                alert('Failed to delete sitemap');
            }
        }
    }

    // Selector Management
    async loadSelectors() {
        await this.loadData();
        this.updateCurrentSitemapDisplay();
        this.renderSelectors();
    }

    updateCurrentSitemapDisplay() {
        const nameElement = document.getElementById('currentSitemapName');
        if (nameElement) {
            nameElement.textContent = this.currentSitemap?.name || 'No sitemap selected';
        }
    }

    renderSelectors() {
        if (!this.currentSitemap || !this.currentSitemap.selectors || this.currentSitemap.selectors.length === 0) {
            this.selectorsGrid.innerHTML = `
                <div class="empty-state">
                    <div class="empty-icon">🎯</div>
                    <h3>No selectors defined</h3>
                    <p>Add selectors to extract data from web pages</p>
                </div>
            `;
            return;
        }

        const selectorsHtml = this.currentSitemap.selectors.map(selector => {
            const typeDisplay = this.getTypeDisplayName(selector.type);
            const multipleIndicator = selector.multiple ? '<span class="multiple-indicator">Multiple</span>' : '';
            
            return `
                <div class="selector-card" data-selector-id="${selector.id}">
                    <div class="selector-header">
                        <div class="selector-id">${this.escapeHtml(selector.id)}</div>
                        <div class="selector-type-info">
                            <span class="selector-type">${typeDisplay}</span>
                            ${multipleIndicator}
                        </div>
                    </div>
                    
                    <div class="selector-css">${this.escapeHtml(selector.selector)}</div>
                    
                    <div class="selector-actions">
                        <button class="btn btn-small btn-secondary" onclick="optionsController.editSelector('${selector.id}')">Edit</button>
                        <button class="btn btn-small btn-secondary" onclick="optionsController.testSelectorOnPage('${selector.id}')">Test</button>
                        <button class="btn btn-small btn-danger" onclick="optionsController.deleteSelector('${selector.id}')">Delete</button>
                    </div>
                </div>
            `;
        }).join('');

        this.selectorsGrid.innerHTML = selectorsHtml;
    }

    openSelectorModal(selectorId = null) {
        if (!this.currentSitemap) {
            alert('Please select a sitemap first');
            return;
        }

        const modal = this.selectorModal;
        const title = document.getElementById('selectorModalTitle');
        const form = this.selectorForm;

        if (selectorId) {
            const selector = this.currentSitemap.selectors.find(s => s.id === selectorId);
            if (selector) {
                title.textContent = 'Edit Selector';
                document.getElementById('selectorId').value = selector.id;
                document.getElementById('selectorType').value = selector.type;
                document.getElementById('selectorCss').value = selector.selector;
                document.getElementById('selectorAttribute').value = selector.attribute || '';
                document.getElementById('selectorMultiple').checked = selector.multiple || false;
                this.handleSelectorTypeChange(selector.type);
                this.currentSelectorId = selectorId;
            }
        } else {
            title.textContent = 'New Selector';
            form.reset();
            document.getElementById('selectorMultiple').checked = false;
            this.handleSelectorTypeChange('text');
            this.currentSelectorId = null;
        }

        modal.classList.add('active');
    }

    closeSelectorModal() {
        this.selectorModal.classList.remove('active');
        this.currentSelectorId = null;
    }

    handleSelectorTypeChange(type) {
        const attributeGroup = document.getElementById('attributeGroup');
        if (attributeGroup) {
            attributeGroup.style.display = type === 'attribute' ? 'block' : 'none';
        }
    }

    async pickSelector() {
        try {
            // Get current active tab
            const [tab] = await chrome.tabs.query({ active: true, currentWindow: true });
            
            // Send message to content script to enable selector mode
            await chrome.tabs.sendMessage(tab.id, { action: 'enableSelector' });
            
            // Close modal temporarily
            this.selectorModal.style.display = 'none';
            
            // Show instructions
            alert('Click on an element on the webpage to select it. Press ESC to cancel.');
        } catch (error) {
            console.error('Error enabling selector mode:', error);
            alert('Failed to enable selector mode. Make sure you have a webpage open.');
        }
    }

    async testSelector() {
        const selectorCss = document.getElementById('selectorCss').value.trim();
        if (!selectorCss) {
            alert('Please enter a CSS selector');
            return;
        }

        try {
            const [tab] = await chrome.tabs.query({ active: true, currentWindow: true });
            const response = await chrome.tabs.sendMessage(tab.id, {
                action: 'testSelector',
                selector: selectorCss
            });

            const testResults = document.getElementById('testResults');
            if (response.elements > 0) {
                testResults.className = 'test-results success';
                testResults.innerHTML = `
                    <strong>✓ Found ${response.elements} element(s)</strong><br>
                    Preview: ${JSON.stringify(response.preview, null, 2)}
                `;
            } else {
                testResults.className = 'test-results error';
                testResults.innerHTML = '<strong>✗ No elements found</strong>';
            }
        } catch (error) {
            console.error('Error testing selector:', error);
            const testResults = document.getElementById('testResults');
            testResults.className = 'test-results error';
            testResults.innerHTML = '<strong>✗ Error testing selector</strong>';
        }
    }

    async saveSelector() {
        const id = document.getElementById('selectorId').value.trim();
        const type = document.getElementById('selectorType').value;
        const css = document.getElementById('selectorCss').value.trim();
        const attribute = document.getElementById('selectorAttribute').value.trim();
        const multiple = document.getElementById('selectorMultiple').checked;

        if (!id || !type || !css) {
            alert('Please fill in all required fields');
            return;
        }

        const selector = {
            id: id,
            type: type,
            selector: css,
            multiple: multiple
        };

        if (type === 'attribute' && attribute) {
            selector.attribute = attribute;
        }

        // Update current sitemap
        if (!this.currentSitemap.selectors) {
            this.currentSitemap.selectors = [];
        }

        const existingIndex = this.currentSitemap.selectors.findIndex(s => s.id === id);
        if (existingIndex >= 0) {
            this.currentSitemap.selectors[existingIndex] = selector;
        } else {
            this.currentSitemap.selectors.push(selector);
        }

        try {
            await chrome.runtime.sendMessage({
                action: 'saveSitemap',
                sitemap: this.currentSitemap
            });

            await chrome.storage.local.set({ currentSitemap: this.currentSitemap });
            
            this.closeSelectorModal();
            this.renderSelectors();
        } catch (error) {
            console.error('Error saving selector:', error);
            alert('Failed to save selector');
        }
    }

    async editSelector(selectorId) {
        this.openSelectorModal(selectorId);
    }

    async deleteSelector(selectorId) {
        if (confirm('Are you sure you want to delete this selector?')) {
            this.currentSitemap.selectors = this.currentSitemap.selectors.filter(s => s.id !== selectorId);
            
            try {
                await chrome.runtime.sendMessage({
                    action: 'saveSitemap',
                    sitemap: this.currentSitemap
                });

                await chrome.storage.local.set({ currentSitemap: this.currentSitemap });
                this.renderSelectors();
            } catch (error) {
                console.error('Error deleting selector:', error);
                alert('Failed to delete selector');
            }
        }
    }

    async selectSitemapForEditing() {
        // Show sitemap selection modal or dropdown
        if (this.sitemaps.length === 0) {
            alert('No sitemaps available. Create a sitemap first.');
            this.switchTab('sitemaps');
            return;
        }

        const sitemapNames = this.sitemaps.map(s => s.name);
        const selectedName = prompt(`Select a sitemap to edit:\n${sitemapNames.map((name, i) => `${i + 1}. ${name}`).join('\n')}\n\nEnter the number:`);
        
        if (selectedName) {
            const index = parseInt(selectedName) - 1;
            if (index >= 0 && index < this.sitemaps.length) {
                this.currentSitemap = this.sitemaps[index];
                await chrome.storage.local.set({ currentSitemap: this.currentSitemap });
                this.updateCurrentSitemapDisplay();
                this.renderSelectors();
            }
        }
    }

    // Settings Management
    async loadSettings() {
        await this.loadData();
        this.populateSettings();
    }

    populateSettings() {
        Object.keys(this.settingsForm).forEach(key => {
            const element = this.settingsForm[key];
            if (element) {
                if (element.type === 'checkbox') {
                    element.checked = this.settings[key];
                } else {
                    element.value = this.settings[key];
                }
            }
        });
    }

    async saveSettings() {
        const newSettings = {};
        
        Object.keys(this.settingsForm).forEach(key => {
            const element = this.settingsForm[key];
            if (element) {
                if (element.type === 'checkbox') {
                    newSettings[key] = element.checked;
                } else if (element.type === 'number') {
                    newSettings[key] = parseInt(element.value);
                } else {
                    newSettings[key] = element.value;
                }
            }
        });

        try {
            await chrome.storage.local.set({ settings: newSettings });
            this.settings = newSettings;
            alert('Settings saved successfully');
        } catch (error) {
            console.error('Error saving settings:', error);
            alert('Failed to save settings');
        }
    }

    async resetSettings() {
        if (confirm('Are you sure you want to reset all settings to defaults?')) {
            this.settings = this.getDefaultSettings();
            await chrome.storage.local.set({ settings: this.settings });
            this.populateSettings();
            alert('Settings reset to defaults');
        }
    }

    // Data Management
    async refreshData() {
        await this.loadData();
        this.renderDataTable();
        this.updateDataStats();
    }

    renderDataTable() {
        if (this.scrapedData.length === 0) {
            this.dataTableHead.innerHTML = '<tr><th>No data available</th></tr>';
            this.dataTableBody.innerHTML = `
                <tr>
                    <td class="empty-data">
                        <div class="empty-icon">📊</div>
                        <p>No scraped data yet. Start scraping to see results here.</p>
                    </td>
                </tr>
            `;
            return;
        }

        // Get all unique keys from data
        const allKeys = new Set();
        this.scrapedData.forEach(item => {
            Object.keys(item).forEach(key => allKeys.add(key));
        });

        const headers = Array.from(allKeys);
        
        // Render table header
        this.dataTableHead.innerHTML = `
            <tr>
                ${headers.map(header => `<th>${this.escapeHtml(header)}</th>`).join('')}
            </tr>
        `;

        // Render table body (first 100 rows for performance)
        const displayData = this.scrapedData.slice(0, 100);
        this.dataTableBody.innerHTML = displayData.map(item => `
            <tr>
                ${headers.map(header => `<td>${this.escapeHtml(String(item[header] || ''))}</td>`).join('')}
            </tr>
        `).join('');
    }

    updateDataStats() {
        document.getElementById('totalRecords').textContent = this.scrapedData.length.toLocaleString();
        document.getElementById('totalSitemaps').textContent = this.sitemaps.length.toLocaleString();
        
        const lastScraped = this.scrapedData.length > 0 ? 
            new Date(Math.max(...this.scrapedData.map(item => new Date(item._scrapedAt || 0).getTime()))).toLocaleDateString() :
            'Never';
        document.getElementById('lastScraped').textContent = lastScraped;
    }

    async clearData() {
        if (confirm('Are you sure you want to clear all scraped data? This cannot be undone.')) {
            try {
                await chrome.runtime.sendMessage({ action: 'clearData' });
                this.scrapedData = [];
                this.renderDataTable();
                this.updateDataStats();
                alert('All data cleared successfully');
            } catch (error) {
                console.error('Error clearing data:', error);
                alert('Failed to clear data');
            }
        }
    }

    filterData() {
        // Implement data filtering logic
        const searchTerm = document.getElementById('searchData')?.value.toLowerCase() || '';
        const selectedSitemap = document.getElementById('filterSitemap')?.value || '';
        
        // This is a simplified version - you could implement more sophisticated filtering
        console.log('Filtering data:', { searchTerm, selectedSitemap });
    }

    // Export Management
    async loadExportOptions() {
        await this.loadData();
        this.populateExportSitemaps();
        this.updateExportPreview();
    }

    populateExportSitemaps() {
        const sitemapSelect = this.exportForm.sitemap;
        if (sitemapSelect) {
            sitemapSelect.innerHTML = '<option value="">All Sitemaps</option>' +
                this.sitemaps.map(sitemap => 
                    `<option value="${sitemap.id}">${this.escapeHtml(sitemap.name)}</option>`
                ).join('');
        }
    }

    updateExportPreview() {
        const previewStats = document.getElementById('exportPreviewStats');
        if (previewStats) {
            // Calculate filtered data count based on current filters
            let filteredCount = this.scrapedData.length;
            
            // Apply filters here if needed
            const selectedSitemap = this.exportForm.sitemap?.value;
            const dateFrom = this.exportForm.dateFrom?.value;
            const dateTo = this.exportForm.dateTo?.value;
            
            previewStats.textContent = `${filteredCount.toLocaleString()} records will be exported`;
        }
    }

    async exportData() {
        const format = Array.from(this.exportForm.format).find(radio => radio.checked)?.value || 'csv';
        
        try {
            const response = await chrome.runtime.sendMessage({
                action: 'exportData',
                format: format
            });

            if (response.success) {
                alert(`Data exported successfully as ${format.toUpperCase()}`);
            } else {
                alert(`Export failed: ${response.error}`);
            }
        } catch (error) {
            console.error('Error exporting data:', error);
            alert('Failed to export data');
        }
    }

    // AI APIs
    async loadAiConfig() {
        try {
            const aiManager = new AIAPIManager();
            const config = await aiManager.loadConfig();
            
            if (this.aiApiElements.apifyApiKey) {
                this.aiApiElements.apifyApiKey.value = config.apify?.apiKey || '';
            }
            if (this.aiApiElements.openrouterApiKey) {
                this.aiApiElements.openrouterApiKey.value = config.openrouter?.apiKey || '';
            }
            if (this.aiApiElements.aiProcessingEnabled) {
                this.aiApiElements.aiProcessingEnabled.checked = config.aiProcessing?.enabled || false;
            }
            if (this.aiApiElements.aiModel) {
                this.aiApiElements.aiModel.value = config.aiProcessing?.model || 'openai/gpt-3.5-turbo';
            }
            if (this.aiApiElements.aiPrompt) {
                this.aiApiElements.aiPrompt.value = config.aiProcessing?.prompt || 'Analyze and summarize the following data:';
            }
            
            this.updateApiStatus();
        } catch (error) {
            console.error('Error loading AI config:', error);
        }
    }

    async saveAiConfig() {
        try {
            const aiManager = new AIAPIManager();
            const config = {
                apify: {
                    apiKey: this.aiApiElements.apifyApiKey?.value || ''
                },
                openrouter: {
                    apiKey: this.aiApiElements.openrouterApiKey?.value || ''
                },
                aiProcessing: {
                    enabled: this.aiApiElements.aiProcessingEnabled?.checked || false,
                    model: this.aiApiElements.aiModel?.value || 'openai/gpt-3.5-turbo',
                    prompt: this.aiApiElements.aiPrompt?.value || 'Analyze and summarize the following data:'
                }
            };
            
            await aiManager.saveConfig(config);
            this.updateApiStatus();
            alert('AI configuration saved successfully!');
        } catch (error) {
            console.error('Error saving AI config:', error);
            alert('Failed to save AI configuration');
        }
    }

    async testApifyConnection() {
        try {
            const aiManager = new AIAPIManager();
            const apiKey = this.aiApiElements.apifyApiKey?.value;
            
            if (!apiKey) {
                alert('Please enter an Apify API key first');
                return;
            }
            
            const result = await aiManager.testApifyConnection(apiKey);
            if (result.success) {
                alert('Apify connection successful!');
                this.updateApiStatus();
            } else {
                alert(`Apify connection failed: ${result.error}`);
            }
        } catch (error) {
            console.error('Error testing Apify connection:', error);
            alert('Failed to test Apify connection');
        }
    }

    async testOpenrouterConnection() {
        try {
            const aiManager = new AIAPIManager();
            const apiKey = this.aiApiElements.openrouterApiKey?.value;
            
            if (!apiKey) {
                alert('Please enter an OpenRouter API key first');
                return;
            }
            
            const result = await aiManager.testOpenrouterConnection(apiKey);
            if (result.success) {
                alert('OpenRouter connection successful!');
                this.updateApiStatus();
            } else {
                alert(`OpenRouter connection failed: ${result.error}`);
            }
        } catch (error) {
            console.error('Error testing OpenRouter connection:', error);
            alert('Failed to test OpenRouter connection');
        }
    }

    async updateApiStatus() {
        try {
            const aiManager = new AIAPIManager();
            const status = await aiManager.getApiStatus();
            
            if (this.aiApiElements.apifyStatus) {
                this.aiApiElements.apifyStatus.textContent = status.apify.configured ? 'Configured' : 'Not configured';
                this.aiApiElements.apifyStatus.className = `status ${status.apify.configured ? 'configured' : 'not-configured'}`;
            }
            
            if (this.aiApiElements.openrouterStatus) {
                this.aiApiElements.openrouterStatus.textContent = status.openrouter.configured ? 'Configured' : 'Not configured';
                this.aiApiElements.openrouterStatus.className = `status ${status.openrouter.configured ? 'configured' : 'not-configured'}`;
            }
        } catch (error) {
            console.error('Error updating API status:', error);
        }
    }

    // About
    loadAbout() {
        const lastUpdatedElement = document.getElementById('lastUpdated');
        if (lastUpdatedElement) {
            lastUpdatedElement.textContent = new Date().toLocaleDateString();
        }
    }

    // Message handling
    handleMessage(message) {
        switch (message.action) {
            case 'selectorSelected':
                // Handle selector picked from content script
                if (this.selectorModal.classList.contains('active')) {
                    document.getElementById('selectorCss').value = message.selector;
                    this.selectorModal.style.display = 'flex';
                }
                break;
        }
    }

    // Utility methods
    getTypeDisplayName(type) {
        const typeNames = {
            'text': 'Text',
            'name': 'Name',
            'email': 'Email',
            'phone': 'Phone Number',
            'url': 'URL',
            'image': 'Image',
            'attribute': 'Attribute',
            'html': 'HTML',
            'number': 'Number',
            'date': 'Date',
            'list': 'List',
            'pagination': 'Pagination',
            'infiniteScroll': 'Infinite Scroll'
        };
        return typeNames[type] || type;
    }

    escapeHtml(text) {
        const div = document.createElement('div');
        div.textContent = text;
        return div.innerHTML;
    }

    closeAllModals() {
        document.querySelectorAll('.modal').forEach(modal => {
            modal.classList.remove('active');
        });
    }
}

// Initialize the options controller
let optionsController;
document.addEventListener('DOMContentLoaded', () => {
    optionsController = new OptionsController();
});