class PopupController {
    constructor() {
        this.isRunning = false;
        this.isPaused = false;
        this.currentSitemap = null;
        this.stats = {
            records: 0,
            pages: 0,
            duplicates: 0
        };
        this.statsInterval = null;
        
        this.initializeElements();
        this.bindEvents();
        this.loadState();
        this.updateUI();
        this.startStatsPolling();
    }

    initializeElements() {
        this.elements = {
            statusDot: document.getElementById('statusDot'),
            statusText: document.getElementById('statusText'),
            recordsCount: document.getElementById('recordsCount'),
            pagesCount: document.getElementById('pagesCount'),
            duplicatesCount: document.getElementById('duplicatesCount'),
            sitemapName: document.getElementById('sitemapName'),
            startBtn: document.getElementById('startBtn'),
            pauseBtn: document.getElementById('pauseBtn'),
            exportCsvBtn: document.getElementById('exportCsvBtn'),
            exportXlsxBtn: document.getElementById('exportXlsxBtn'),
            optionsBtn: document.getElementById('optionsBtn')
        };
    }

    bindEvents() {
        this.elements.startBtn.addEventListener('click', () => this.startScraping());
        this.elements.pauseBtn = document.getElementById('pauseBtn');
        this.elements.stopBtn = document.getElementById('stopBtn');
        this.elements.pauseBtn.addEventListener('click', () => this.togglePause());
        this.elements.stopBtn.addEventListener('click', () => this.stopScraping());
        this.elements.exportCsvBtn.addEventListener('click', () => this.exportData('csv'));
        this.elements.exportXlsxBtn.addEventListener('click', () => this.exportData('xlsx'));
        this.elements.optionsBtn.addEventListener('click', () => this.openOptions());

        // Listen for updates from background script
        chrome.runtime.onMessage.addListener((message, sender, sendResponse) => {
            this.handleMessage(message);
        });
    }

    async loadState() {
        try {
            const result = await chrome.storage.local.get([
                'scrapingState',
                'currentSitemap',
                'scrapingStats'
            ]);

            this.isRunning = result.scrapingState?.isRunning || false;
            this.currentSitemap = result.currentSitemap || null;
            this.stats = result.scrapingStats || this.stats;
        } catch (error) {
            console.error('Error loading state:', error);
        }
    }

    updateUI() {
        // Update status
        if (this.isRunning) {
            this.elements.statusDot.className = 'status-dot running';
            this.elements.statusText.textContent = 'Running';
            this.elements.startBtn.disabled = true;
            this.elements.pauseBtn.disabled = false;
        } else {
            this.elements.statusDot.className = 'status-dot';
            this.elements.statusText.textContent = 'Ready';
            this.elements.startBtn.disabled = false;
            this.elements.pauseBtn.disabled = true;
        }

        // Update stats with real-time data
        this.updateStatsDisplay();

        // Update sitemap name
        this.elements.sitemapName.textContent = this.currentSitemap?.name || 'None selected';

        // Enable/disable export buttons based on data availability
        const hasData = this.stats.records > 0;
        this.elements.exportCsvBtn.disabled = !hasData;
        this.elements.exportXlsxBtn.disabled = !hasData;
    }

    updateStatsDisplay() {
        // Update basic stats
        this.elements.recordsCount.textContent = this.stats.records.toLocaleString();
        this.elements.pagesCount.textContent = this.stats.pages.toLocaleString();
        this.elements.duplicatesCount.textContent = this.stats.duplicates.toLocaleString();
        
        // Update additional stats if elements exist
        const errorCountEl = document.getElementById('errorCount');
        const runtimeEl = document.getElementById('runtime');
        const recordsPerMinuteEl = document.getElementById('recordsPerMinute');
        
        if (errorCountEl) errorCountEl.textContent = this.stats.errors || 0;
        if (runtimeEl) runtimeEl.textContent = this.stats.runtimeFormatted || '0s';
        if (recordsPerMinuteEl) recordsPerMinuteEl.textContent = (this.stats.recordsPerMinute || 0) + '/min';
        
        // Update progress section visibility
        const progressSection = document.getElementById('progressSection');
        if (progressSection) {
            progressSection.style.display = this.isRunning ? 'block' : 'none';
        }
    }

    async startScraping() {
        if (!this.currentSitemap) {
            this.showError('Please select a sitemap in the Options page first.');
            return;
        }

        try {
            await chrome.runtime.sendMessage({
                type: 'startScraping',
                sitemap: this.currentSitemap
            });
            
            this.isRunning = true;
            this.updateUI();
        } catch (error) {
            console.error('Error starting scraping:', error);
            this.showError('Failed to start scraping. Please try again.');
        }
    }

    togglePause() {
        chrome.runtime.sendMessage({ type: 'pauseScraping' });
    }

    stopScraping() {
        chrome.runtime.sendMessage({ type: 'stopScraping' });
    }



    async exportData(format) {
        try {
            const response = await chrome.runtime.sendMessage({
                type: 'exportData',
                format: format
            });
            
            if (response && response.success) {
                this.showSuccess('Data exported successfully!');
            } else {
                this.showError('Export failed: ' + (response?.error || 'Unknown error'));
            }
        } catch (error) {
            console.error('Error exporting data:', error);
            this.showError('Failed to export data.');
        }
    }

    openOptions() {
        chrome.runtime.openOptionsPage();
    }

    startStatsPolling() {
        // Poll for stats every 1 second when popup is open
        this.statsInterval = setInterval(() => {
            try {
                chrome.runtime.sendMessage({ type: 'getStats' }, (response) => {
                    if (chrome.runtime.lastError) {
                        console.warn('Stats polling error:', chrome.runtime.lastError);
                        return;
                    }
                    if (response && response.stats) {
                        this.updateStatsDisplay(response.stats);
                        this.updateButtonStates(response.isRunning, response.isPaused);
                        this.updateProgressDisplay(response.progress);
                    }
                });
            } catch (error) {
                console.error('Error in stats polling:', error);
            }
        }, 1000);
    }

    stopStatsPolling() {
        if (this.statsInterval) {
            clearInterval(this.statsInterval);
            this.statsInterval = null;
        }
    }

    updateProgressDisplay(progress) {
        const progressSection = document.querySelector('.progress-section');
        const progressFill = document.querySelector('.progress-fill');
        const progressText = document.querySelector('.progress-text');
        
        if (progress && progressSection) {
            progressSection.style.display = 'block';
            if (progressFill) {
                progressFill.style.width = `${progress.percentage || 0}%`;
            }
            if (progressText) {
                progressText.textContent = progress.text || 'Processing...';
            }
        } else if (progressSection) {
            progressSection.style.display = 'none';
        }
    }

    handleMessage(message) {
        switch (message.type) {
            case 'statsUpdated':
                this.updateStatsDisplay(message.stats);
                this.updateButtonStates(message.isRunning, message.isPaused);
                this.updateProgressDisplay(message.progress);
                break;
            case 'scrapingStarted':
                this.isRunning = true;
                this.isPaused = false;
                this.updateButtonStates(true, false);
                break;
            case 'scrapingPaused':
                this.isPaused = true;
                this.updateButtonStates(true, true);
                break;
            case 'scrapingResumed':
                this.isPaused = false;
                this.updateButtonStates(true, false);
                break;
            case 'scrapingStopped':
                this.isRunning = false;
                this.isPaused = false;
                this.updateButtonStates(false, false);
                break;
            case 'scrapingCompleted':
                this.isRunning = false;
                this.isPaused = false;
                this.updateButtonStates(false, false);
                this.showSuccess('Scraping completed successfully!');
                break;
            case 'scrapingStateChanged':
                this.isRunning = message.isRunning;
                this.updateUI();
                break;
            case 'sitemapChanged':
                this.currentSitemap = message.sitemap;
                this.updateUI();
                break;
            case 'error':
                this.showError(message.message);
                break;
            case 'success':
                this.showSuccess(message.message);
                break;
        }
    }

    updateButtonStates(isRunning, isPaused) {
        const startBtn = this.elements.startBtn;
        const pauseBtn = this.elements.pauseBtn;
        const stopBtn = document.getElementById('stopBtn');
        
        if (isRunning) {
            startBtn.style.display = 'none';
            pauseBtn.style.display = 'inline-block';
            if (stopBtn) stopBtn.style.display = 'inline-block';
            
            if (isPaused) {
                pauseBtn.textContent = 'Resume';
            } else {
                pauseBtn.textContent = 'Pause';
            }
        } else {
            startBtn.style.display = 'inline-block';
            pauseBtn.style.display = 'none';
            if (stopBtn) stopBtn.style.display = 'none';
        }
    }

    showError(message) {
        this.elements.statusDot.className = 'status-dot error';
        this.elements.statusText.textContent = 'Error';
        
        // Show error notification (you could implement a toast system here)
        console.error('Popup Error:', message);
        
        // Reset status after 3 seconds
        setTimeout(() => {
            this.elements.statusDot.className = this.isRunning ? 'status-dot running' : 'status-dot';
            this.elements.statusText.textContent = this.isRunning ? 'Running' : 'Ready';
        }, 3000);
    }

    showSuccess(message) {
        // Show success notification
        console.log('Popup Success:', message);
    }
}

// Initialize popup when DOM is loaded
document.addEventListener('DOMContentLoaded', () => {
    const popup = new PopupController();
    
    // Cleanup when popup is closed
    window.addEventListener('beforeunload', () => {
        popup.stopStatsPolling();
    });
});