<?php
/**
 * Plugin Name: Airwave Audio Player Matrix
 * Plugin URI: https://spunwebtechnology.com/airwaveautomator/
 * Description: Radio streaming player with Matrix chat integration for WordPress. Features audio streaming, Matrix chat, and customizable widgets.
 * Version: 0.0.96
 * Author: Spun Web Technology
 * Author URI: https://spunwebtechnology.com
 * License: MIT
 * Text Domain: airwave-audio-player-matrix
 * Domain Path: /languages
 */

// Prevent direct access
if (!defined('ABSPATH')) {
    exit;
}

// Define plugin constants
if (!defined('AIRWAVE_PLUGIN_URL')) { define('AIRWAVE_PLUGIN_URL', plugin_dir_url(__FILE__)); }
if (!defined('AIRWAVE_PLUGIN_PATH')) { define('AIRWAVE_PLUGIN_PATH', plugin_dir_path(__FILE__)); }
// Define plugin version
if (!defined('AIRWAVE_PLUGIN_VERSION')) {
    define('AIRWAVE_PLUGIN_VERSION', '0.0.96');
}

// Force WordPress to recognize the new version
add_action('init', function() {
    // Clear any cached plugin data
    if (function_exists('wp_clean_plugins_cache')) {
        wp_clean_plugins_cache();
    }
    
    // Force refresh of plugin headers
    $plugin_file = plugin_dir_path(__FILE__) . 'airwave-audio-player-matrix.php';
    if (file_exists($plugin_file)) {
        $plugin_data = get_plugin_data($plugin_file);
        // This forces WordPress to re-read the plugin header
    }
}, 1);

// Include required files
require_once AIRWAVE_PLUGIN_PATH . 'includes/admin.php';
require_once AIRWAVE_PLUGIN_PATH . 'includes/frontend.php';
require_once AIRWAVE_PLUGIN_PATH . 'includes/shortcodes.php';
require_once AIRWAVE_PLUGIN_PATH . 'includes/widgets.php';
require_once AIRWAVE_PLUGIN_PATH . 'includes/meta-proxy.php';

// Note: Plugin updates are now handled by the "Spun Plugin Updater" plugin
// Install and activate "Spun Plugin Updater" to enable automatic updates
// Configure it in Plugin Updater → Manage Updates with:
// - Plugin File: airwave-audio-player-matrix/airwave-audio-player-matrix.php
// - Plugin Slug: airwave-audio-player-matrix
// - Repository URL: https://spunwebtechnology.com/wp-content/uploads/repository/airwave-audio-player-matrix/
// - Version Constant: AIRWAVE_PLUGIN_VERSION
// Repository Info: https://spunwebtechnology.com/wp-content/uploads/repository/airwave-audio-player-matrix/info.json

if (!class_exists('AirwaveAudioPlayerMatrix')) { class AirwaveAudioPlayerMatrix {
    
    public function __construct() {
        add_action('init', array($this, 'init'));
        add_action('wp_enqueue_scripts', array($this, 'enqueue_scripts'));
        add_action('admin_enqueue_scripts', array($this, 'enqueue_admin_scripts'));
        
        // Register shortcodes (legacy/awmx variants)
        add_shortcode('airwave_radio', array($this, 'radio_shortcode'));
        add_shortcode('matrix_chat', array($this, 'matrix_shortcode'));
        
        // Ensure AWMX handler overrides any legacy registration
        add_action('init', array($this, 'override_radio_shortcode'), 1000);
        
        // AJAX handlers
        add_action('wp_ajax_airwave_resolve_stream', array($this, 'resolve_stream'));
        add_action('wp_ajax_airwave_meta', array($this, 'get_metadata'));
        add_action('wp_ajax_nopriv_airwave_resolve_stream', array($this, 'resolve_stream'));
        add_action('wp_ajax_nopriv_airwave_meta', array($this, 'get_metadata'));
    }
    
    public function init() {
        // Initialize admin or frontend with guards to prevent fatal errors
        if (is_admin()) {
            if (!class_exists('AirwaveAdmin')) {
                $admin_file = AIRWAVE_PLUGIN_PATH . 'includes/admin.php';
                if (file_exists($admin_file)) {
                    require_once $admin_file;
                }
            }
            if (class_exists('AirwaveAdmin')) {
                new AirwaveAdmin();
            } else {
                error_log('[Airwave] AirwaveAdmin missing; skipping admin init');
            }
        } else {
            if (!class_exists('AirwaveFrontend')) {
                $front_file = AIRWAVE_PLUGIN_PATH . 'includes/frontend.php';
                if (file_exists($front_file)) {
                    require_once $front_file;
                }
            }
            if (class_exists('AirwaveFrontend')) {
                new AirwaveFrontend();
            }
        }
        
        // Ensure Airwave shortcodes (new combined/matrix/radio variants) are registered
        // This instantiates the class that calls add_shortcode for airwave_combined and others.
        if (class_exists('AirwaveShortcodes')) {
            new AirwaveShortcodes();
        }
    }
    
    public function override_radio_shortcode() {
        // Remove any prior bindings to ensure AWMX UI is used
        remove_shortcode('airwave_radio');
        add_shortcode('airwave_radio', array($this, 'radio_shortcode'));
    }
    
    public function enqueue_scripts() {
        // Enqueue CSS
        wp_enqueue_style(
            'awmx-css',
            set_url_scheme(AIRWAVE_PLUGIN_URL . 'assets/css/awmx.css'),
            array(),
            AIRWAVE_PLUGIN_VERSION
        );
        
        // Enqueue JavaScript
        wp_enqueue_script(
            'awmx-audio',
            set_url_scheme(AIRWAVE_PLUGIN_URL . 'assets/js/awmx-audio.js'),
            array(),
            AIRWAVE_PLUGIN_VERSION,
            true
        );
        
        wp_enqueue_script(
            'awmx-meta',
            set_url_scheme(AIRWAVE_PLUGIN_URL . 'assets/js/awmx-meta.js'),
            array(),
            AIRWAVE_PLUGIN_VERSION,
            true
        );
        
        wp_enqueue_script(
            'awmx-matrix',
            set_url_scheme(AIRWAVE_PLUGIN_URL . 'assets/js/awmx-matrix.js'),
            array(),
            AIRWAVE_PLUGIN_VERSION,
            true
        );
        
        // Enqueue Matrix SDK
        wp_enqueue_script(
            'matrix-sdk',
            set_url_scheme(AIRWAVE_PLUGIN_URL . 'assets/js/matrix-sdk.min.js'),
            array(),
            AIRWAVE_PLUGIN_VERSION,
            true
        );
        
        // Localize script with settings
        $settings = get_option('airwave_settings', array());
        $radio_settings = array(
            'mode' => $settings['radio_mode'] ?? 'single',
            'rotate_interval' => intval($settings['rotate_interval'] ?? 60),
            'autoplay' => !empty($settings['autoplay']),
            'streams' => array()
        );
        
        // Build streams array
        for ($i = 1; $i <= 5; $i++) {
            $stream_key = "stream_$i";
            if (!empty($settings["{$stream_key}_url"])) {
                $radio_settings['streams'][] = array(
                    'name' => $settings["{$stream_key}_name"] ?? "Stream $i",
                    'type' => $settings["{$stream_key}_type"] ?? 'shoutcast',
                    'url' => $settings["{$stream_key}_url"],
                    'meta_url' => !empty($settings["{$stream_key}_meta_url"]) ? $settings["{$stream_key}_meta_url"] : ($settings["{$stream_key}_api_url"] ?? ''),
                    'sonic_port' => $settings["{$stream_key}_sonic_port"] ?? ''
                );
            }
        }
        
        // If Sonic Panel is enabled and no streams are configured, add one from Sonic settings
        if (empty($radio_settings['streams']) && !empty($settings['sonic_enabled']) && !empty($settings['sonic_stream_url'])) {
            $radio_settings['streams'][] = array(
                'name' => 'Sonic Panel',
                'type' => 'sonicpanel',
                'url' => $settings['sonic_stream_url'],
                'meta_url' => $settings['sonic_json_api'] ?? '',
                'sonic_port' => $settings['sonic_port'] ?? ''
            );
        }
        $matrix_settings = array(
            'homeserver' => $settings['matrix_hs'] ?? 'https://matrix-client.matrix.org',
            'room' => $settings['matrix_room'] ?? ''
        );
        
        // Get default cover image URL if set
        $default_cover_id = absint($settings['default_cover_image'] ?? 0);
        $default_cover_url = $default_cover_id ? wp_get_attachment_image_url($default_cover_id, 'medium') : '';
        
        wp_localize_script('awmx-audio', 'AWMX', array(
            'ajax' => admin_url('admin-ajax.php', is_ssl() ? 'https' : 'http'),
            'nonce' => wp_create_nonce('airwave_nonce'),
            'radio' => $radio_settings,
            'matrix' => $matrix_settings,
            'defaultCoverUrl' => $default_cover_url ?: '' // URL to custom default cover image, empty string if not set
        ));
    }
    
    public function enqueue_admin_scripts($hook) {
        if ($hook === 'settings_page_airwave-settings') {
            wp_enqueue_style(
                'awmx-admin-css',
                AIRWAVE_PLUGIN_URL . 'assets/css/awmx.css',
                array(),
                AIRWAVE_PLUGIN_VERSION
            );
        }
    }
    
    // RADIO SHORTCODE
    public function radio_shortcode($atts) {
        $atts = shortcode_atts(array(
            'title' => 'Radio Station',
            'theme' => 'light',
            'stream' => '0',
            'show_history' => 'false'
        ), $atts);
        
        $theme = ($atts['theme'] === 'dark') ? 'dark' : 'light';
        $stream_index = intval($atts['stream']);
        $show_history = ($atts['show_history'] === 'true' || $atts['show_history'] === '1');
        
        ob_start();
        ?>
        <div class="awmx-card awmx-theme-<?php echo $theme; ?>" data-stream="<?php echo esc_attr($stream_index); ?>" data-show-history="<?php echo $show_history ? 'true' : 'false'; ?>">
            <div class="awmx-card__header">
                <h3><?php echo esc_html($atts['title']); ?></h3>
            </div>
            <div class="awmx-radio__body">
                <div class="awmx-controls">
                    <button class="awmx-btn awmx-play">▶ Start</button>
                    <button class="awmx-btn awmx-pause" style="display:none;">⏸ Pause</button>
                    <button class="awmx-btn awmx-stop">⏹ Stop</button>
                    <div class="awmx-inline">
                        <label>Volume:</label>
                        <input type="range" class="awmx-vol" min="0" max="1" step="0.1" value="0.8">
                    </div>
                    <div class="awmx-inline">
                        <label>Stream:</label>
                        <select class="awmx-stream-select"></select>
                    </div>
                </div>
                
                <div class="awmx-warn" hidden></div>
                
                <div class="awmx-meta-grid">
                    <?php 
                    // Placeholder image as base64 SVG to prevent layout shift
                    $placeholder = 'data:image/svg+xml;base64,PHN2ZyB3aWR0aD0iMjAwIiBoZWlnaHQ9IjIwMCIgeG1sbnM9Imh0dHA6Ly93d3cudzMub3JnLzIwMDAvc3ZnIj48cmVjdCB3aWR0aD0iMjAwIiBoZWlnaHQ9IjIwMCIgZmlsbD0iIzMzMzMzMyIvPjx0ZXh0IHg9IjUwJSIgeT0iNTAlIiBmb250LWZhbWlseT0iQXJpYWwiIGZvbnQtc2l6ZT0iMTQiIGZpbGw9IiM4ODg4ODgiIHRleHQtYW5jaG9yPSJtaWRkbGUiIGR5PSIuM2VtIj5ObyBDb3ZlcjwvdGV4dD48L3N2Zz4=';
                    ?>
                    <img id="awmx_art" src="<?php echo esc_attr($placeholder); ?>" alt="Album Cover" style="display:inline; width:200px; height:200px; object-fit:cover; background:#333;">
                    <div class="awmx-stack awmx-meta-inline">
                        <div class="awmx-now-playing-container">
                            <strong>Now Playing:</strong>
                            <div class="awmx-title-marquee">
                                <span id="awmx_title" class="awmx-title-text">–</span>
                            </div>
                        </div>
                        <div><strong>Listeners:</strong> <span id="awmx_listeners">–</span></div>
                        <div><strong>DJ:</strong> <span id="awmx_dj">–</span></div>
                    </div>
                </div>
                
                <?php if ($show_history): ?>
                <div class="awmx-hint">Recent songs:</div>
                <div id="awmx_history" class="awmx-stack"></div>
                <?php endif; ?>
                
                <div class="awmx-viz-container">
                    <canvas class="awmx-viz-canvas" width="400" height="120"></canvas>
                </div>
                
                <audio class="awmx-audio" preload="none" crossorigin="anonymous"></audio>
            </div>
        </div>
        <?php
        return ob_get_clean();
    }
    
    // MATRIX SHORTCODE
    public function matrix_shortcode($atts) {
        $atts = shortcode_atts(array(
            'title' => 'Matrix Chat',
            'homeserver' => '',
            'room' => '',
            'theme' => 'light'
        ), $atts);
        
        $settings = get_option('airwave_settings', array());
        $homeserver = $atts['homeserver'] ?: ($settings['matrix_hs'] ?? 'https://matrix-client.matrix.org');
        $room = $atts['room'] ?: ($settings['matrix_room'] ?? '');
        $theme = ($atts['theme'] === 'dark') ? 'dark' : 'light';
        
        ob_start();
        ?>
        <div class="awmx-card awmx-theme-<?php echo $theme; ?>">
            <div class="awmx-card__header">
                <h3><?php echo esc_html($atts['title']); ?></h3>
            </div>
            <div class="awmx-matrix__root" data-homeserver="<?php echo esc_attr($homeserver); ?>" data-room="<?php echo esc_attr($room); ?>">
                
                <!-- Login Form -->
                <form class="awmx-matrix__login">
                    <div class="awmx-stack">
                        <div class="awmx-inline">
                            <label>Homeserver:</label>
                            <input type="url" name="homeserver" value="<?php echo esc_attr($homeserver); ?>" placeholder="https://matrix-client.matrix.org">
                        </div>
                        <div class="awmx-inline">
                            <label>Username:</label>
                            <input type="text" name="username" placeholder="@user:domain.com" required>
                        </div>
                        <div class="awmx-inline">
                            <label>Password:</label>
                            <input type="password" name="password" required>
                        </div>
                        <button type="submit" class="awmx-btn">Login</button>
                    </div>
                </form>
                
                <!-- Chat App -->
                <div class="awmx-matrix__app" hidden>
                    <div class="awmx-matrix__sidebar">
                        <div class="awmx-room__header">
                            <h4>Rooms</h4>
                        </div>
                        <div class="awmx-room__list"></div>
                        
                        <div class="awmx-stack">
                            <div class="awmx-inline">
                                <input type="text" class="awmx-join-input" placeholder="#room:domain.com">
                                <button class="awmx-btn awmx-join-btn">Join</button>
                            </div>
                        </div>
                        
                        <button class="awmx-btn awmx-logout">Logout</button>
                    </div>
                    
                    <div class="awmx-matrix__timeline">
                        <div class="awmx-room__header">
                            <span class="awmx-room__name">Select a room</span>
                        </div>
                        <div class="awmx-events"></div>
                        <form class="awmx-composer">
                            <input type="text" class="awmx-input" placeholder="Type your message...">
                            <button type="submit" class="awmx-btn">Send</button>
                        </form>
                    </div>
                </div>
            </div>
        </div>
        <?php
        return ob_get_clean();
    }
    
    // AJAX: Resolve stream URL (handle playlists/redirects)
    public function resolve_stream() {
        check_ajax_referer('airwave_nonce', 'nonce');
        
        $url = sanitize_url($_POST['url'] ?? '');
        if (empty($url)) {
            wp_send_json_error('No URL provided');
        }
        
        // Handle common playlist formats
        $context = stream_context_create(array(
            'http' => array(
                'timeout' => 10,
                'user_agent' => 'Airwave/1.0',
                'follow_location' => true,
                'max_redirects' => 5
            )
        ));
        
        try {
            $response = wp_remote_head($url, array(
                'timeout' => 10,
                'redirection' => 5,
                'user-agent' => 'Airwave Audio Player Matrix/' . AIRWAVE_PLUGIN_VERSION,
            ));

            if (is_wp_error($response)) {
                // Retry with relaxed SSL verification to avoid handshake issues
                $response = wp_remote_head($url, array(
                    'timeout' => 10,
                    'redirection' => 5,
                    'sslverify' => false,
                    'user-agent' => 'Airwave Audio Player Matrix/' . AIRWAVE_PLUGIN_VERSION,
                ));
            }

            if (is_wp_error($response)) {
                wp_send_json_error('Cannot resolve stream');
            }

            $content_type = wp_remote_retrieve_header($response, 'content-type');
            $content_type = is_array($content_type) ? end($content_type) : ($content_type ?? '');

            // If content-type is missing, probe with GET
            if (empty($content_type)) {
                $probe = wp_remote_get($url, array(
                    'timeout' => 10,
                    'redirection' => 5,
                    'headers' => array('Accept' => 'audio/*, application/*'),
                    'user-agent' => 'Airwave Audio Player Matrix/' . AIRWAVE_PLUGIN_VERSION,
                ));
                if (!is_wp_error($probe)) {
                    $ct = wp_remote_retrieve_header($probe, 'content-type');
                    $content_type = is_array($ct) ? end($ct) : ($ct ?? '');
                }
            }

            // Handle M3U playlists
            if (
                stripos($content_type, 'audio/x-mpegurl') !== false ||
                stripos($content_type, 'application/vnd.apple.mpegurl') !== false ||
                stripos($content_type, 'application/x-mpegurl') !== false ||
                stripos($content_type, 'audio/mpegurl') !== false
            ) {
                $playlist_res = wp_remote_get($url, array(
                    'timeout' => 10,
                    'redirection' => 5,
                    'user-agent' => 'Airwave Audio Player Matrix/' . AIRWAVE_PLUGIN_VERSION,
                    // Fallback to avoid SSL handshake issues on some hosts
                    'sslverify' => false,
                ));

                if (is_wp_error($playlist_res)) {
                    wp_send_json_error('Failed to fetch playlist');
                }

                $content = wp_remote_retrieve_body($playlist_res);
                if ($content) {
                    $lines = preg_split('/\r\n|\r|\n/', $content);
                    foreach ($lines as $line) {
                        $line = trim($line);
                        if ($line && $line[0] !== '#') {
                            wp_send_json_success(array('url' => $line));
                        }
                    }
                }
                wp_send_json_error('Empty playlist');
            }

            // Not a playlist, return original URL
            wp_send_json_success(array('url' => $url));
        } catch (Exception $e) {
            wp_send_json_error('Error: ' . $e->getMessage());
        }
    }
    
    // AJAX: Get metadata
    public function get_metadata() {
        check_ajax_referer('airwave_nonce', 'nonce');
        
        $type = sanitize_text_field($_REQUEST['type'] ?? 'shoutcast');
        $base_url = sanitize_url($_REQUEST['base'] ?? '');
        $meta_url = sanitize_url($_REQUEST['meta_url'] ?? '');
        $sonic_port = sanitize_text_field($_REQUEST['sonic_port'] ?? '');
        
        // Heuristic: derive SonicPanel details from base URL when missing
        $derived_from_base = false;
        if (!empty($base_url)) {
            $host = parse_url($base_url, PHP_URL_HOST);
            $path = parse_url($base_url, PHP_URL_PATH) ?? '';
            if (empty($sonic_port) && is_string($path) && preg_match('/\/(\d{2,6})\/stream\b/i', $path, $m)) {
                $sonic_port = $m[1];
                $derived_from_base = true;
            }
            if (empty($meta_url) && !empty($host) && !empty($sonic_port)) {
                // Prefer SonicPanel JSON that accepts port as query param - always use HTTPS
                $meta_url = sprintf('https://%s/cp/get_info.php?p=%s', $host, $sonic_port);
                $derived_from_base = true;
            }
            if ($type !== 'sonicpanel' && (!empty($sonic_port) || (is_string($meta_url) && strpos($meta_url, '/cp/get_info.php') !== false))) {
                $type = 'sonicpanel';
            }
        }
        
        // Auto-override with global Sonic Panel settings when stream lacks config
        $settings = get_option('airwave_settings', array());
        $sonic_enabled = !empty($settings['sonic_enabled']);
        $global_meta_url = sanitize_url($settings['sonic_json_api'] ?? '');
        $global_sonic_port = sanitize_text_field($settings['sonic_port'] ?? '');
        $override_applied = false;
        if ($sonic_enabled) {
            if (empty($meta_url) && !empty($global_meta_url)) {
                $meta_url = $global_meta_url;
                $override_applied = true;
            }
            if (empty($sonic_port) && !empty($global_sonic_port)) {
                $sonic_port = $global_sonic_port;
                $override_applied = true;
            }
            // If Sonic signals present, ensure type is sonicpanel
            if ($type !== 'sonicpanel' && (!empty($sonic_port) || (is_string($meta_url) && strpos($meta_url, '/cp/get_info.php') !== false))) {
                $type = 'sonicpanel';
                $override_applied = true;
            }
        }
        
        // Debug: Add parameters to response for debugging
        $debug_info = array(
            'type' => $type,
            'base_url' => $base_url,
            'meta_url' => $meta_url,
            'sonic_port' => $sonic_port,
            'global_sonic_enabled' => $sonic_enabled,
            'override_applied' => $override_applied,
            'derived_from_base' => $derived_from_base
        );
        
        $meta_proxy = new AirwaveMetaProxy();
        $result = $meta_proxy->get_metadata($type, $base_url, $meta_url, $sonic_port);
        
        // Add debug info and marker to result
        if (is_array($result)) {
            $result['debug'] = $debug_info;
            $result['response_marker'] = 'awmx_debug_v1';
        }
        
        if ($result) {
            wp_send_json_success($result);
        } else {
            wp_send_json_error('Failed to fetch metadata');
        }
    }
}

// Initialize the plugin
new AirwaveAudioPlayerMatrix();

}

// Ensure a published test page exists with radio and chat shortcodes
add_action('init', function() {
    if (!is_admin()) {
        return;
    }
    if (!current_user_can('manage_options')) {
        return;
    }
    $title = 'Airwave Test';
    $slug = 'airwave-test';
    $content = '[airwave_radio title="Radio Player"]\n[matrix_chat title="Matrix Chat"]';

    // Try to locate existing page by path/slug
    $existing = get_page_by_path($slug, OBJECT, 'page');
    if ($existing && $existing instanceof WP_Post) {
        // If content is missing expected shortcodes, refresh it
        $needs_update = empty($existing->post_content)
            || (strpos($existing->post_content, '[airwave_radio') === false)
            || (strpos($existing->post_content, '[matrix_chat') === false);

        if ($needs_update) {
            wp_update_post(array(
                'ID'           => $existing->ID,
                'post_title'   => $title,
                'post_name'    => $slug,
                'post_content' => $content,
                'post_status'  => 'publish',
                'post_type'    => 'page'
            ));
        }
        return;
    }

    // Create the page if it doesn't exist
    wp_insert_post(array(
        'post_title'   => $title,
        'post_name'    => $slug,
        'post_content' => $content,
        'post_status'  => 'publish',
        'post_type'    => 'page'
    ));
});