(function(){
    const q=(s,r=document)=>r.querySelector(s);
    const qa=(s,r=document)=>Array.from(r.querySelectorAll(s));
    let rotateTimer=null, metaTimer=null;

    function sameScheme(url){
        try{
            const urlProtocol = new URL(url).protocol;
            const pageProtocol = location.protocol;
            
            // Allow HTTPS streams on HTTP sites (safe for audio streaming)
            // Only block HTTP streams on HTTPS sites (actual mixed content security issue)
            if (pageProtocol === 'https:' && urlProtocol === 'http:') {
                return false; // Block HTTP on HTTPS (security risk)
            }
            
            // Special case: if the stream URL is HTTPS but the page tries to load it as HTTP,
            // upgrade the page protocol or allow it (some servers redirect HTTP to HTTPS)
            if (pageProtocol === 'http:' && urlProtocol === 'https:') {
                return true; // Allow HTTPS on HTTP page (server may handle redirect)
            }
            
            return true; // Allow all other combinations
        }catch{
            return true;
        }
    }

    async function resolveStream(url){
        // Force HTTPS if page is HTTPS to avoid mixed content
        let resolvedUrl=url;
        if(location.protocol==='https:' && url.startsWith('http://')){
            resolvedUrl=url.replace(/^http:\/\//i,'https://');
        }
        const fd=new FormData();
        fd.append('action','airwave_resolve_stream');
        fd.append('nonce',AWMX.nonce);
        fd.append('url',resolvedUrl);
        try{
            const res=await fetch(AWMX.ajax,{method:'POST',body:fd,credentials:'same-origin'});
            const j=await res.json();
            let finalUrl=(j&&j.success&&j.data&&j.data.url)?j.data.url:resolvedUrl;
            // Ensure final URL uses HTTPS if page is HTTPS
            if(location.protocol==='https:' && finalUrl.startsWith('http://')){
                finalUrl=finalUrl.replace(/^http:\/\//i,'https://');
            }
            return finalUrl;
        }catch(e){
            console.warn('Stream resolution failed, using converted URL:',e);
            return resolvedUrl;
        }
    }

    function inferType(stream){
        const blob=((stream&&stream.meta_url)||'')+' '+((stream&&stream.url)||'');
        if(/get_info\.php|sonic/i.test(blob))return 'sonicpanel';
        if(/icecast/i.test(blob))return 'icecast';
        if(/shoutcast|7\.html|stats/i.test(blob))return 'shoutcast';
        return 'sonicpanel'; // default
    }

    async function pollMeta(type, base, metaUrl, sonicPort, stream){
        // Infer type if missing
        let inferredType=type||inferType(stream);
        
        const u=new URL(AWMX.ajax);
        u.searchParams.set('action','airwave_meta');
        u.searchParams.set('nonce',AWMX.nonce);
        u.searchParams.set('type',inferredType);
        u.searchParams.set('base',base);
        if(metaUrl)u.searchParams.set('meta_url',metaUrl);
        if(sonicPort)u.searchParams.set('sonic_port',sonicPort);
        try{
            const r=await fetch(u.toString(),{credentials:'same-origin'});
            const j=await r.json();
            if(!j.success)return;
            const d=j.data||{};
            if(q('#awmx_title')){
                const titleEl=q('#awmx_title');
                const titleText=d.title||'';
                titleEl.textContent=titleText;
                // Enable scrolling marquee if text overflows
                const marquee=q('.awmx-title-marquee');
                if(marquee){
                    checkMarquee(titleEl, marquee);
                }
            }
            q('#awmx_listeners')&&(q('#awmx_listeners').textContent=d.listeners??'');
            q('#awmx_dj')&&(q('#awmx_dj').textContent=d.dj||'');
            if(q('#awmx_art')){
                const artEl=q('#awmx_art');
                // Use custom default cover or placeholder if no art, keep image visible to maintain layout
                const placeholder='data:image/svg+xml;base64,PHN2ZyB3aWR0aD0iMjAwIiBoZWlnaHQ9IjIwMCIgeG1sbnM9Imh0dHA6Ly93d3cudzMub3JnLzIwMDAvc3ZnIj48cmVjdCB3aWR0aD0iMjAwIiBoZWlnaHQ9IjIwMCIgZmlsbD0iIzMzMzMzMyIvPjx0ZXh0IHg9IjUwJSIgeT0iNTAlIiBmb250LWZhbWlseT0iQXJpYWwiIGZvbnQtc2l6ZT0iMTQiIGZpbGw9IiM4ODg4ODgiIHRleHQtYW5jaG9yPSJtaWRkbGUiIGR5PSIuM2VtIj5ObyBDb3ZlcjwvdGV4dD48L3N2Zz4=';
                const defaultCover=AWMX?.defaultCoverUrl||'';
                if(d.art){
                    artEl.src=d.art;
                    artEl.style.display='inline';
                    artEl.style.visibility='visible';
                } else {
                    // Use custom default cover if set, otherwise use placeholder
                    const fallbackImage=defaultCover||placeholder;
                    if(!artEl.src||artEl.src===placeholder||artEl.src===''||(!defaultCover&&artEl.src===defaultCover)){
                        artEl.src=fallbackImage;
                    }
                    artEl.style.display='inline';
                    artEl.style.visibility='visible';
                }
                // Ensure image has fixed dimensions to prevent layout shift
                if(!artEl.style.width&&!artEl.style.height){
                    artEl.style.width='200px';
                    artEl.style.height='200px';
                    artEl.style.objectFit='cover';
                }
            }
            if(q('#awmx_history')&&Array.isArray(d.history)){
                q('#awmx_history').innerHTML=d.history.slice(0,20).map(x=>`<div>${String(x).replace(/[<>&]/g,'')}</div>`).join('');
            }
        }catch{}
    }

    async function playStream(section, stream){
        if(!stream || !stream.url){
            console.error('playStream: Invalid stream object',stream);
            const warn=q('.awmx-warn',section);
            if(warn){
                warn.hidden=false;
                warn.textContent='No stream URL configured.';
            }
            return;
        }
        
        const audio=q('.awmx-audio',section);
        if(!audio){
            console.error('playStream: Audio element not found in section');
            return;
        }
        
        const warn=q('.awmx-warn',section);
        const state=section.__awmx||(section.__awmx={reconnectAttempts:0,reconnectTimer:null,progressTimer:null,lastTimeUpdate:0,userPaused:false,currentStream:null,currentDirect:null});
        state.currentStream=stream;
        state.reconnectAttempts=0;
        if(state.reconnectTimer){clearTimeout(state.reconnectTimer);state.reconnectTimer=null;}
        if(state.progressTimer){clearInterval(state.progressTimer);state.progressTimer=null;}

        // Try to resolve and test the stream
        if(warn){
            warn.hidden=true;
            warn.textContent='';
        }
        
        // Force HTTPS if page is HTTPS to avoid mixed content
        let streamUrl = stream.url;
        if(location.protocol==='https:' && streamUrl.startsWith('http://')){
            streamUrl=streamUrl.replace(/^http:\/\//i,'https://');
        }
        
        const direct=await resolveStream(streamUrl);
        state.currentDirect=direct;
        
        // Configure audio element properties
        audio.crossOrigin='anonymous';
        audio.playsInline=true;
        // Set volume to default if not set
        if(audio.volume === 0 || isNaN(audio.volume)){
            audio.volume = 0.8;
        }
        audio.muted = false;
        
        const updateButtonStates = () => {
            const playBtn=q('.awmx-play',section);
            const pauseBtn=q('.awmx-pause',section);
            const stopBtn=q('.awmx-stop',section);
            if(playBtn) playBtn.style.display='none';
            if(pauseBtn) pauseBtn.style.display='inline-block';
            if(stopBtn) stopBtn.style.display='inline-block';
            if(warn) {
                warn.hidden=true;
                warn.textContent='';
            }
        };
        
        // Play function - matches v0.0.66 pattern
        const playNow = async () => {
            try {
                const playPromise = audio.play();
                if(playPromise !== undefined) {
                    await playPromise;
                    console.log('Audio playback started successfully');
                    updateButtonStates();
                }
            } catch(err) {
                console.error('Playback error:', err);
                if(warn){
                    warn.hidden=false;
                    warn.textContent='Failed to play stream: '+(err.message||'Browser prevented playback');
                }
            }
        };
        
        // Match v0.0.66 logic exactly: Check if we need to load new stream
        // Condition: readyState === 0 (not loaded) AND src is different
        const needsLoad = (audio.readyState === 0 && audio.src !== direct);
        const isSameStream = (audio.src === direct && audio.src !== '');
        
        console.log('Setting up audio playback:', {
            src: direct,
            currentSrc: audio.src,
            readyState: audio.readyState,
            needsLoad: needsLoad,
            isSameStream: isSameStream,
            volume: audio.volume,
            muted: audio.muted
        });
        
        if(needsLoad) {
            // New stream needs to load - set source, load, and wait for canplay (like v0.0.66)
            audio.src = direct;
            audio.load();
            
            // Wait for the audio to be ready to play (matches v0.0.66 exactly)
            const canplayHandler = async function onCanPlay() {
                audio.removeEventListener('canplay', onCanPlay);
                console.log('canplay event fired, readyState:', audio.readyState);
                await playNow();
            };
            
            audio.addEventListener('canplay', canplayHandler, { once: true });
            
            // Handle load errors (from v0.0.66)
            const errorHandler = function onError() {
                audio.removeEventListener('error', onError);
                console.error('Audio load error');
                if(warn){
                    warn.hidden=false;
                    warn.textContent='Failed to load stream. Please check the stream URL.';
                }
            };
            audio.addEventListener('error', errorHandler, { once: true });
            
        } else if(isSameStream) {
            // Same stream already loaded - just resume playback (matches v0.0.66)
            console.log('Same stream, resuming playback');
            await playNow();
        } else {
            // Different stream that's already loaded - need to reload
            audio.pause();
            audio.src = direct;
            audio.load();
            
            const canplayHandler = async function onCanPlay() {
                audio.removeEventListener('canplay', onCanPlay);
                console.log('canplay event fired after reload, readyState:', audio.readyState);
                await playNow();
            };
            
            audio.addEventListener('canplay', canplayHandler, { once: true });
            
            const errorHandler = function onError() {
                audio.removeEventListener('error', onError);
                console.error('Audio load error');
                if(warn){
                    warn.hidden=false;
                    warn.textContent='Failed to load stream. Please check the stream URL.';
                }
            };
            audio.addEventListener('error', errorHandler, { once: true });
        }
        
        if(metaTimer) clearInterval(metaTimer);
        const tick=()=>pollMeta(stream.type||'shoutcast',direct,stream.meta_url||'',stream.sonic_port||'',stream);
        tick();
        metaTimer=setInterval(tick,15000);
        
        // Setup visualizer if canvas exists (simplified like v0.0.76)
        if(section){
            const canvas=q('canvas.awmx-viz-canvas',section)||q('canvas',section);
            if(canvas && (window.AudioContext||window.webkitAudioContext)){
                try{
                    // Only setup once per canvas
                    if(canvas.dataset.vizSetup) return;
                    canvas.dataset.vizSetup='1';
                    
                    const ctx=new (window.AudioContext||window.webkitAudioContext)();
                    const analyser=ctx.createAnalyser();
                    analyser.fftSize=256;
                    const dataArray=new Uint8Array(analyser.frequencyBinCount);
                    const source=ctx.createMediaElementSource(audio);
                    source.connect(analyser);
                    analyser.connect(ctx.destination);
                    const canvasCtx=canvas.getContext('2d');
                    
                    function draw(){
                        if(audio.paused||audio.ended||!audio.src){
                            // Clear canvas when paused
                            canvasCtx.fillStyle='#1a1a1a';
                            canvasCtx.fillRect(0,0,canvas.width,canvas.height);
                            requestAnimationFrame(draw);
                            return;
                        }
                        analyser.getByteFrequencyData(dataArray);
                        canvasCtx.fillStyle='#1a1a1a';
                        canvasCtx.fillRect(0,0,canvas.width,canvas.height);
                        const barWidth=(canvas.width/dataArray.length)*2.5;
                        let x=0;
                        for(let i=0;i<dataArray.length;i++){
                            const barHeight=(dataArray[i]/255)*canvas.height;
                            const gradient=canvasCtx.createLinearGradient(0,canvas.height-barHeight,0,canvas.height);
                            gradient.addColorStop(0,'#4a9eff');
                            gradient.addColorStop(1,'#1e3a8a');
                            canvasCtx.fillStyle=gradient;
                            canvasCtx.fillRect(x,canvas.height-barHeight,barWidth,barHeight);
                            x+=barWidth+1;
                        }
                        requestAnimationFrame(draw);
                    }
                    draw();
                }catch(e){
                    console.warn('Visualizer setup failed:',e);
                }
            }
        }
    }

    function bind(section){
        console.log('bind() called for section:',section);
        const audio=q('.awmx-audio',section);
        const playBtn=q('.awmx-play',section);
        const pauseBtn=q('.awmx-pause',section);
        const stopBtn=q('.awmx-stop',section);
        const vol=q('.awmx-vol',section);
        const warn=q('.awmx-warn',section);
        
        console.log('Elements found:',{
            audio:!!audio,
            playBtn:!!playBtn,
            pauseBtn:!!pauseBtn,
            stopBtn:!!stopBtn,
            vol:!!vol,
            warn:!!warn
        });
        
        console.log('AWMX:',typeof AWMX!=='undefined'?AWMX:'NOT DEFINED');
        console.log('AWMX.radio:',AWMX?.radio);
        console.log('AWMX.radio.streams:',AWMX?.radio?.streams);
        
        const streams=Array.isArray(AWMX.radio?.streams)?AWMX.radio.streams.filter(s=>s&&s.url):[];
        console.log('Filtered streams:',streams);
        if(!streams.length){
            if(warn){
                warn.hidden=false;
                warn.textContent='No streams configured. Set them in Settings → Airwave Radio + Matrix.';
            }
            return;
        }
        let idx=0;
        const playCurrent=async()=>{
            console.log('playCurrent called, idx:',idx,'stream:',streams[idx]);
            await playStream(section,streams[idx]);
        };
        // Only rotate if mode is 'rotate' AND there are multiple streams
        if(AWMX.radio?.mode==='rotate' && streams.length>1){
            const ms=Math.max(10000,(AWMX.radio.rotate_interval||60)*1000);
            if(rotateTimer) clearInterval(rotateTimer);
            rotateTimer=setInterval(()=>{
                idx=(idx+1)%streams.length;
                playCurrent();
            }, ms);
        }
        // State holder per section
        const state=section.__awmx||(section.__awmx={reconnectAttempts:0,reconnectTimer:null,progressTimer:null,lastTimeUpdate:0,userPaused:false,currentStream:null,currentDirect:null});

        const scheduleReconnect=(reason)=>{
            if(state.userPaused) return;
            if(state.reconnectTimer) return;
            const delay=Math.min(30000,1000*Math.pow(2,state.reconnectAttempts||0));
            state.reconnectAttempts=(state.reconnectAttempts||0)+1;
            if(warn){
                warn.hidden=false;
                warn.textContent=`Reconnecting… (${reason}) in ${Math.round(delay/1000)}s`;
            }
            state.reconnectTimer=setTimeout(async()=>{
                try{
                    const s=state.currentStream||streams[idx];
                    const direct=await resolveStream(s.url).catch(()=>state.currentDirect||s.url);
                    state.currentDirect=direct;
                    audio.src=direct;
                    audio.load();
                    if(!state.userPaused){
                        try{ await audio.play(); }catch{}
                    }
                } finally {
                    state.reconnectTimer=null;
                }
            }, delay);
        };

        const onPlaying=()=>{
            state.reconnectAttempts=0;
            if(warn){ warn.hidden=true; warn.textContent=''; }
            if(state.progressTimer) clearInterval(state.progressTimer);
            state.lastTimeUpdate=Date.now();
            state.progressTimer=setInterval(()=>{
                if(audio.paused) return;
                const stale=Date.now()-state.lastTimeUpdate;
                if(stale>10000){ // no progress for 10s
                    scheduleReconnect('stalled');
                }
            },5000);
        };
        const onPause=()=>{
            if(state.progressTimer){ clearInterval(state.progressTimer); state.progressTimer=null; }
        };
        const onTimeUpdate=()=>{ state.lastTimeUpdate=Date.now(); };
        const onError=()=>{ scheduleReconnect('error'); };
        const onStalled=()=>{ scheduleReconnect('stalled'); };
        const onWaiting=()=>{ // if buffering for >5s, attempt reconnect
            setTimeout(()=>{ if(!audio.paused && audio.readyState<2) scheduleReconnect('buffering'); },5000);
        };

        // Stream selector - hide when only 1 stream, show and populate when multiple
        const streamSelect=q('.awmx-stream-select',section);
        const streamSelectContainer=streamSelect?streamSelect.closest('.awmx-inline'):null;
        if(streamSelect){
            if(streams.length>1){
                // Show selector and populate it
                if(streamSelectContainer) streamSelectContainer.style.display='';
                streamSelect.innerHTML='';
                streams.forEach((s,i)=>{
                    const opt=document.createElement('option');
                    opt.value=i;
                    opt.textContent=s.name||`Stream ${i+1}`;
                    if(i===idx) opt.selected=true;
                    streamSelect.appendChild(opt);
                });
                streamSelect.addEventListener('change',(e)=>{
                    idx=parseInt(e.target.value)||0;
                    if(!audio.paused) playCurrent();
                });
            } else {
                // Hide selector when only 1 stream
                if(streamSelectContainer) streamSelectContainer.style.display='none';
            }
        }

        // Only auto-play if autoplay is enabled
        if(AWMX.radio?.autoplay){
            playCurrent();
        }

        playBtn&&playBtn.addEventListener('click',async()=>{ 
            console.log('Play button clicked, idx:',idx,'streams:',streams);
            state.userPaused=false;
            if(streamSelect&&streamSelect.value!==''){
                idx=parseInt(streamSelect.value)||0;
            }
            if(!streams[idx]){
                console.error('No stream at index',idx);
                const warn=q('.awmx-warn',section);
                if(warn){
                    warn.hidden=false;
                    warn.textContent='No stream configured at index '+idx;
                }
                return;
            }
            console.log('Calling playStream with stream:',streams[idx]);
            await playCurrent();
        });
        pauseBtn&&pauseBtn.addEventListener('click',()=>{ 
            state.userPaused=true; 
            audio.pause(); 
            if(playBtn) playBtn.style.display='inline-block';
            if(pauseBtn) pauseBtn.style.display='none';
        });
        stopBtn&&stopBtn.addEventListener('click',()=>{ 
            state.userPaused=true;
            audio.pause();
            audio.src='';
            audio.load();
            state.currentStream=null;
            state.currentDirect=null;
            if(playBtn) playBtn.style.display='inline-block';
            if(pauseBtn) pauseBtn.style.display='none';
            if(stopBtn) stopBtn.style.display='inline-block';
            // Clear metadata
            const titleEl=q('#awmx_title',section);
            const listenersEl=q('#awmx_listeners',section);
            const djEl=q('#awmx_dj',section);
            const artEl=q('#awmx_art',section);
            if(titleEl) titleEl.textContent='–';
            if(listenersEl) listenersEl.textContent='–';
            if(djEl) djEl.textContent='–';
            // Reset to custom default cover or placeholder instead of hiding - maintains layout
            if(artEl) {
                const placeholder='data:image/svg+xml;base64,PHN2ZyB3aWR0aD0iMjAwIiBoZWlnaHQ9IjIwMCIgeG1sbnM9Imh0dHA6Ly93d3cudzMub3JnLzIwMDAvc3ZnIj48cmVjdCB3aWR0aD0iMjAwIiBoZWlnaHQ9IjIwMCIgZmlsbD0iIzMzMzMzMyIvPjx0ZXh0IHg9IjUwJSIgeT0iNTAlIiBmb250LWZhbWlseT0iQXJpYWwiIGZvbnQtc2l6ZT0iMTQiIGZpbGw9IiM4ODg4ODgiIHRleHQtYW5jaG9yPSJtaWRkbGUiIGR5PSIuM2VtIj5ObyBDb3ZlcjwvdGV4dD48L3N2Zz4=';
                const defaultCover=AWMX?.defaultCoverUrl||'';
                artEl.src=defaultCover||placeholder;
                artEl.style.display='inline';
                artEl.style.visibility='visible';
            }
            if(metaTimer) clearInterval(metaTimer);
            metaTimer=null;
        });
        vol&&vol.addEventListener('input',e=>audio.volume=parseFloat(e.target.value||'1'));

        // Attach audio event listeners
        audio.addEventListener('playing',onPlaying);
        audio.addEventListener('pause',onPause);
        audio.addEventListener('timeupdate',onTimeUpdate);
        audio.addEventListener('error',onError);
        audio.addEventListener('stalled',onStalled);
        audio.addEventListener('waiting',onWaiting);
        audio.addEventListener('emptied',()=>scheduleReconnect('emptied'));
        audio.addEventListener('ended',()=>scheduleReconnect('ended'));

        // Attempt to recover when connectivity returns
        window.addEventListener('online',()=>{
            if(state.userPaused) return;
            scheduleReconnect('online');
        });
    }

    // Tab switching for combined shortcode
    function initTabs(){
        const tabs=qa('.airwave-tabs');
        tabs.forEach(tabContainer=>{
            const tabBtns=qa('.airwave-tab-btn',tabContainer);
            const tabPanels=qa('.airwave-tab-panel',tabContainer);
            tabBtns.forEach(btn=>{
                btn.addEventListener('click',()=>{
                    const targetTab=btn.dataset.tab;
                    // Update button states
                    tabBtns.forEach(b=>b.classList.remove('active'));
                    btn.classList.add('active');
                    // Update panel states
                    tabPanels.forEach(p=>p.classList.remove('active'));
                    const targetPanel=q('#'+targetTab,tabContainer);
                    if(targetPanel){
                        targetPanel.classList.add('active');
                    }
                });
            });
        });
    }

    // Initialize visualizer shortcode
    function initVisualizerShortcode(container){
        const streamIdx=parseInt(container.dataset.stream)||0;
        const canvas=q('canvas',container)||q('.airwave-viz-canvas',container);
        const audio=q('.airwave-audio-player',container)||q('audio',container);
        if(!canvas||!audio) return;
        
        const streams=AWMX?.radio?.streams||[];
        if(!streams[streamIdx]){
            console.warn('Visualizer: Stream index',streamIdx,'not found');
            return;
        }
        
        const stream=streams[streamIdx];
        let vizContext=null,analyser=null,dataArray=null,animationFrame=null;
        
        // Auto-play and setup visualizer
        (async()=>{
            try{
                // Resolve stream
                let streamUrl=stream.url;
                if(location.protocol==='https:'&&streamUrl.startsWith('http://')){
                    streamUrl=streamUrl.replace(/^http:\/\//i,'https://');
                }
                const direct=await resolveStream(streamUrl);
                
                // Setup audio
                audio.crossOrigin='anonymous';
                audio.playsInline=true;
                audio.volume=0.8;
                audio.muted=false;
                audio.src=direct;
                audio.load();
                
                // Play audio
                try{
                    await audio.play();
                }catch(e){
                    console.warn('Visualizer autoplay blocked:',e);
                }
                
                // Setup visualizer
                if(window.AudioContext||window.webkitAudioContext){
                    vizContext=new (window.AudioContext||window.webkitAudioContext)();
                    analyser=vizContext.createAnalyser();
                    analyser.fftSize=256;
                    dataArray=new Uint8Array(analyser.frequencyBinCount);
                    const source=vizContext.createMediaElementSource(audio);
                    source.connect(analyser);
                    analyser.connect(vizContext.destination);
                    
                    const ctx=canvas.getContext('2d');
                    function draw(){
                        if(audio.paused||audio.ended||!audio.src){
                            if(animationFrame) cancelAnimationFrame(animationFrame);
                            return;
                        }
                        analyser.getByteFrequencyData(dataArray);
                        ctx.fillStyle='#1a1a1a';
                        ctx.fillRect(0,0,canvas.width,canvas.height);
                        const barWidth=(canvas.width/dataArray.length)*2.5;
                        let x=0;
                        for(let i=0;i<dataArray.length;i++){
                            const barHeight=(dataArray[i]/255)*canvas.height;
                            const gradient=ctx.createLinearGradient(0,canvas.height-barHeight,0,canvas.height);
                            gradient.addColorStop(0,'#4a9eff');
                            gradient.addColorStop(1,'#1e3a8a');
                            ctx.fillStyle=gradient;
                            ctx.fillRect(x,canvas.height-barHeight,barWidth,barHeight);
                            x+=barWidth+1;
                        }
                        animationFrame=requestAnimationFrame(draw);
                    }
                    draw();
                }
            }catch(err){
                console.error('Visualizer initialization failed:',err);
            }
        })();
    }
    
    // Initialize song ticker shortcode
    function initSongTicker(container){
        const streamIdx=parseInt(container.dataset.stream)||0;
        const limit=parseInt(container.dataset.limit)||10;
        const scrollSpeed=parseInt(container.dataset.scrollSpeed)||3;
        const titleEl=q('.airwave-song-title',container);
        const artistEl=q('.airwave-song-artist',container);
        const artEl=q('.airwave-album-art',container);
        const tickerTrack=q('.airwave-ticker-track',container);
        
        const streams=AWMX?.radio?.streams||[];
        if(!streams[streamIdx]){
            console.warn('Song ticker: Stream index',streamIdx,'not found');
            if(titleEl) titleEl.textContent='No stream configured';
            if(tickerTrack) tickerTrack.innerHTML='<span class="airwave-history-item">No stream configured</span>';
            return;
        }
        
        const stream=streams[streamIdx];
        let metaTimer=null;
        let animationFrame=null;
        
        // Fetch metadata and update display
        async function updateTicker(){
            try{
                const type=(stream.type&&stream.type.trim())?stream.type:inferType(stream);
                const ajaxUrl=AWMX?.ajax||(typeof airwaveAjax!=='undefined'?airwaveAjax.ajaxUrl:'/wp-admin/admin-ajax.php');
                const nonce=AWMX?.nonce||(typeof airwaveAjax!=='undefined'?airwaveAjax.nonce:'');
                
                const payload=new URLSearchParams({
                    action:'airwave_meta',
                    nonce:nonce,
                    type:type||'',
                    base:stream.url||'',
                    meta_url:stream.meta_url||'',
                    sonic_port:stream.sonic_port||''
                });
                
                const resp=await fetch(ajaxUrl,{
                    method:'POST',
                    credentials:'same-origin',
                    headers:{'Content-Type':'application/x-www-form-urlencoded'},
                    body:payload.toString()
                });
                
                const json=await resp.json();
                if(json&&json.success){
                    const d=json.data||{};
                    
                    // Update current song
                    if(titleEl){
                        const combo=(d.artist?(d.artist+' — '):'')+(d.title||'');
                        titleEl.textContent=combo.trim()||'No song playing';
                    }
                    if(artistEl){
                        artistEl.textContent=d.artist||'';
                    }
                    
                    // Update album art
                    if(artEl){
                        if(d.art){
                            artEl.src=d.art;
                            artEl.style.display='block';
                        }else{
                            artEl.style.display='none';
                        }
                    }
                    
                    // Update history ticker
                    if(tickerTrack){
                        if(Array.isArray(d.history)&&d.history.length>0){
                            // Stop existing animation
                            if(animationFrame){
                                cancelAnimationFrame(animationFrame);
                                animationFrame=null;
                            }
                            
                            const items=d.history.slice(0,limit).map(h=>{
                                const t=String(h).replace(/</g,'&lt;').replace(/>/g,'&gt;').replace(/&/g,'&amp;');
                                return`<span class="airwave-history-item">${t}</span>`;
                            });
                            tickerTrack.innerHTML=items.join(' • ');
                            tickerTrack.style.display='block';
                            tickerTrack.style.whiteSpace='nowrap';
                            
                            // Wait for DOM to update, then check if scrolling needed
                            setTimeout(()=>{
                                if(tickerTrack.scrollWidth>tickerTrack.parentElement.clientWidth){
                                    let pos=0;
                                    const animate=()=>{
                                        pos-=scrollSpeed;
                                        if(pos<=-tickerTrack.scrollWidth){
                                            pos=tickerTrack.parentElement.clientWidth;
                                        }
                                        tickerTrack.style.transform=`translateX(${pos}px)`;
                                        tickerTrack.style.transition='none';
                                        animationFrame=requestAnimationFrame(animate);
                                    };
                                    animate();
                                }else{
                                    tickerTrack.style.transform='translateX(0)';
                                }
                            },100);
                        }else{
                            tickerTrack.innerHTML='<span class="airwave-history-item">No recent songs</span>';
                            tickerTrack.style.transform='translateX(0)';
                            if(animationFrame){
                                cancelAnimationFrame(animationFrame);
                                animationFrame=null;
                            }
                        }
                    }
                }else{
                    console.warn('Song ticker metadata fetch failed:',json);
                    if(titleEl) titleEl.textContent='Unable to fetch metadata';
                }
            }catch(err){
                console.error('Song ticker update failed:',err);
                if(titleEl) titleEl.textContent='Error loading data';
            }
        }
        
        // Update immediately and set interval
        updateTicker();
        if(metaTimer) clearInterval(metaTimer);
        metaTimer=setInterval(updateTicker,15000);
    }

    // Function to check and enable marquee for a title element
    function checkMarquee(titleEl, marquee){
        if(!titleEl || !marquee) {
            console.log('checkMarquee: Missing elements', {titleEl: !!titleEl, marquee: !!marquee});
            return;
        }
        
        // Force a reflow to ensure accurate measurements
        void titleEl.offsetWidth;
        void marquee.offsetWidth;
        
        setTimeout(()=>{
            // Get actual text width by temporarily allowing it to expand
            const tempStyle = {
                position: titleEl.style.position,
                visibility: titleEl.style.visibility,
                display: titleEl.style.display
            };
            titleEl.style.position = 'absolute';
            titleEl.style.visibility = 'hidden';
            titleEl.style.display = 'inline-block';
            titleEl.style.width = 'auto';
            titleEl.style.whiteSpace = 'nowrap';
            
            const textWidth = titleEl.scrollWidth || titleEl.offsetWidth;
            
            // Restore original styles
            titleEl.style.position = tempStyle.position || '';
            titleEl.style.visibility = tempStyle.visibility || '';
            titleEl.style.display = tempStyle.display || '';
            
            // Get container width
            const containerWidth = marquee.getBoundingClientRect().width || marquee.clientWidth || marquee.offsetWidth;
            
            console.log('Marquee check:', {
                textWidth: textWidth,
                containerWidth: containerWidth,
                text: titleEl.textContent.substring(0, 50),
                needsScroll: textWidth > containerWidth,
                textContent: titleEl.textContent
            });
            
            if(textWidth > containerWidth && containerWidth > 0 && textWidth > 0){
                const scrollDistance = textWidth - containerWidth + 100; // Extra padding for smooth loop
                const duration = Math.max(8, Math.min(30, (scrollDistance / 20))); // 20px per second
                
                // Remove any existing transform
                titleEl.style.transform = '';
                
                marquee.classList.add('scrolling');
                marquee.style.setProperty('--scroll-duration', duration + 's');
                marquee.style.setProperty('--container-width', containerWidth + 'px');
                marquee.style.setProperty('--scroll-distance', scrollDistance + 'px');
                
                // Force browser to recognize the animation
                void marquee.offsetWidth;
                void titleEl.offsetWidth;
                
                console.log('Marquee ENABLED:', {
                    scrollDistance: scrollDistance + 'px',
                    duration: duration + 's',
                    animation: 'awmx-scroll-marquee-loop',
                    hasClass: marquee.classList.contains('scrolling'),
                    cssVars: {
                        duration: marquee.style.getPropertyValue('--scroll-duration'),
                        distance: marquee.style.getPropertyValue('--scroll-distance')
                    }
                });
            }else{
                marquee.classList.remove('scrolling');
                titleEl.style.transform = 'translateX(0)';
                console.log('Marquee DISABLED - text fits in container', {
                    textWidth,
                    containerWidth,
                    condition: textWidth > containerWidth && containerWidth > 0 && textWidth > 0
                });
            }
        }, 300); // Increased timeout to ensure DOM is ready
    }

    document.addEventListener('DOMContentLoaded',()=>{
        console.log('DOMContentLoaded - Finding .awmx-radio__body elements');
        const bodies=qa('.awmx-radio__body');
        console.log('Found',bodies.length,'.awmx-radio__body elements');
        bodies.forEach((body,i)=>{
            console.log('Binding section',i,body);
            bind(body);
            
            // Check for marquee on initial load
            setTimeout(()=>{
                const titleEl=q('#awmx_title',body);
                const marquee=q('.awmx-title-marquee',body);
                if(titleEl && marquee){
                    checkMarquee(titleEl, marquee);
                    // Also check after a longer delay to catch dynamic content
                    setTimeout(()=>checkMarquee(titleEl, marquee), 1000);
                }
            }, 500);
        });
        
        // Initialize tabs for combined shortcode
        initTabs();
        
        // Initialize visualizer shortcodes
        qa('.airwave-visualizer-shortcode').forEach(initVisualizerShortcode);
        
        // Initialize song ticker shortcodes
        qa('.airwave-song-ticker-shortcode').forEach(initSongTicker);
    });
})();

