/* global matrixcs */
(function(){
    const q=(s,r=document)=>r.querySelector(s);
    const qa=(s,r=document)=>Array.from(r.querySelectorAll(s));
    const LS_KEY='awmx_matrix_session_v2';
    
    function renderEvent(e){
        try{
            const c=e.getContent();
            if(e.getType()==='m.room.message'&&c&&c.body){
                const ts=new Date(e.getTs()).toLocaleTimeString();
                const div=document.createElement('div');
                div.className='awmx-msg';
                const sender=document.createElement('strong');
                sender.className='awmx-sender';
                sender.textContent=e.getSender();
                const time=document.createElement('span');
                time.className='awmx-ts';
                time.textContent=ts+' ';
                const body=document.createElement('span');
                body.className='awmx-body';
                body.textContent=c.body;
                div.appendChild(time);
                div.appendChild(sender);
                div.append(':\u00A0');
                div.appendChild(body);
                return div;
            }
        }catch{}
        return null;
    }
    
    function mount(root){
        const loginForm=q('.awmx-matrix__login',root);
        const app=q('.awmx-matrix__app',root);
        const joinInput=q('.awmx-join-input',root);
        const joinBtn=q('.awmx-join-btn',root);
        const eventsEl=q('.awmx-events',root);
        const composer=q('.awmx-composer',root);
        const composerIn=q('.awmx-input',root);
        const roomListEl=q('.awmx-room__list',root);
        const roomNameEl=q('.awmx-room__name',root);
        const logoutBtn=q('.awmx-logout',root);
        const defaultHS=root.dataset.homeserver||AWMX?.matrix?.homeserver||'https://matrix-client.matrix.org';
        const defaultRoom=root.dataset.room||AWMX?.matrix?.room||'';
        let client=null, currentRoomId=null;
        
        const setUI=(on)=>{
            loginForm.hidden=!!on;
            app.hidden=!on;
        };
        
        const save=(s)=>localStorage.setItem(LS_KEY,JSON.stringify(s));
        const load=()=>{
            try{
                return JSON.parse(localStorage.getItem(LS_KEY)||'null');
            }catch{return null;}
        };
        const clear=()=>localStorage.removeItem(LS_KEY);
        
        function createClient(hs,token,userId,deviceId){
            return matrixcs.createClient({
                baseUrl:hs,
                accessToken:token,
                userId,
                deviceId
            });
        }

        async function tryRefreshSession(saved){
            const hs=saved?.hs; const rt=saved?.refreshToken; const at=saved?.accessToken; const uid=saved?.userId; const did=saved?.deviceId;
            if(!hs||!rt) return false;
            try{
                const resp=await fetch(`${hs.replace(/\/$/, '')}/_matrix/client/v3/refresh`,{
                    method:'POST',
                    headers:{'Content-Type':'application/json'},
                    body:JSON.stringify({ refresh_token: rt })
                });
                if(!resp.ok) throw new Error(`Refresh failed: ${resp.status}`);
                const j=await resp.json();
                const newAccess=j.access_token||at; const newRefresh=j.refresh_token||rt;
                save({ hs, userId:uid, accessToken:newAccess, refreshToken:newRefresh, deviceId:did });
                try{ client && client.stopClient(); }catch{}
                client=createClient(hs,newAccess,uid,did);
                startClient();
                return true;
            }catch(err){
                console.warn('Matrix token refresh failed:', err);
                return false;
            }
        }
        
        function onSyncPrepared(){
            const rooms=client.getRooms().filter(r=>r&&(r.name||r.roomId)).sort((a,b)=>(a.name||'').localeCompare(b.name||''));
            roomListEl.innerHTML='';
            rooms.forEach(r=>{
                const b=document.createElement('button');
                b.className='awmx-room__btn';
                b.dataset.roomid=r.roomId;
                b.textContent=r.name||r.roomId;
                b.addEventListener('click',()=>openRoomId(r.roomId));
                roomListEl.appendChild(b);
            });
            if(defaultRoom) openByAlias(defaultRoom);
        }
        
        function startClient(){
            client.startClient({initialSyncLimit:30});
            client.once('sync',(state)=>{
                if(state==='PREPARED') onSyncPrepared();
            });
            client.on('Room.timeline',(event,room)=>{
                if(!room||room.roomId!==currentRoomId) return;
                const line=renderEvent(event);
                if(line){
                    eventsEl.appendChild(line);
                    eventsEl.scrollTop=eventsEl.scrollHeight;
                }
            });
            client.on('sync',(state,prev,data)=>{
                if(state==='RECONNECTING'){
                    // optional: could show a status indicator
                    // console.log('Matrix reconnecting…');
                } else if(state==='ERROR'){
                    const http=(data&&data.error&& (data.error.httpStatus||data.error.statusCode))||0;
                    const code=(data&&data.error&&data.error.errcode)||'';
                    if(http===401||code==='M_UNKNOWN_TOKEN'||code==='M_FORBIDDEN'){
                        const saved=load();
                        (async()=>{
                            const refreshed=await tryRefreshSession(saved);
                            if(!refreshed){
                                try{ client && client.stopClient(); }catch{}
                                clear();
                                setUI(false);
                                alert('Your Matrix session has expired. Please sign in again.');
                            }
                        })();
                    }
                }
            });
        }
        
        async function openRoomId(roomId){
            currentRoomId=roomId;
            const room=client.getRoom(roomId);
            roomNameEl.textContent=room?.name||roomId;
            eventsEl.innerHTML='';
            const tl=room?.getLiveTimeline();
            if(tl){
                tl.getEvents().slice(-50).forEach(e=>{
                    const line=renderEvent(e);
                    if(line) eventsEl.appendChild(line);
                });
                eventsEl.scrollTop=eventsEl.scrollHeight;
            }
        }
        
        async function openByAlias(alias){
            try{
                const res=await client.joinRoom(alias);
                openRoomId(res.roomId||res);
            }catch(e){
                const room=client.getRooms().find(r=> r.getAliases?.().includes(alias));
                if(room) openRoomId(room.roomId);
            }
        }
        
        // Prevent default form submission
        loginForm.setAttribute('onsubmit', 'return false;');
        
        loginForm.addEventListener('submit', async (e)=>{
            e.preventDefault();
            e.stopPropagation();
            const fd=new FormData(loginForm);
            const hs=(fd.get('homeserver')||defaultHS).toString().trim();
            const uid=(fd.get('username')||'').toString().trim();
            const pwd=(fd.get('password')||'').toString();
            if(!uid||!pwd){
                alert('Please enter both username and password');
                return;
            }
            const btn=loginForm.querySelector('button[type=submit]');
            const origText=btn.textContent;
            btn.disabled=true;
            btn.textContent='Logging in...';
            try{
                const temp=matrixcs.createClient({baseUrl:hs});
                const resp=await temp.login('m.login.password',{
                    identifier:{type:'m.id.user',user:uid},
                    password:pwd,
                    initial_device_display_name:'AirwaveWP'
                });
                client=createClient(hs,resp.access_token,resp.user_id,resp.device_id);
                save({hs,userId:resp.user_id,accessToken:resp.access_token,refreshToken:resp.refresh_token||null,deviceId:resp.device_id});
                setUI(true);
                startClient();
                // Clear password field
                loginForm.querySelector('.awmx-password').value='';
            }catch(err){
                console.error('Matrix login error:',err);
                alert('Matrix login failed: '+(err?.message||err));
            } finally {
                btn.disabled=false;
                btn.textContent=origText;
            }
        });
        
        (function(){
            const s=load();
            if(!s) return;
            try{
                client=createClient(s.hs,s.accessToken,s.userId,s.deviceId);
                setUI(true);
                startClient();
            }catch{
                clear();
            }
        })();
        
        joinBtn.addEventListener('click',(e)=>{
            e.preventDefault();
            const alias=joinInput.value.trim();
            if(alias) openByAlias(alias);
        });
        
        composer.addEventListener('submit', async (e)=>{
            e.preventDefault();
            const text=composerIn.value.trim();
            if(!text||!currentRoomId) return;
            try{
                await client.sendTextMessage(currentRoomId,text);
                composerIn.value='';
            }catch(err){
                alert('Send failed: '+(err?.message||err));
            }
        });
        
        logoutBtn.addEventListener('click', async ()=>{
            try{
                try{
                    await client.logout();
                }catch{}
            } finally {
                clear();
                location.reload();
            }
        });

        // Restart sync when connectivity returns
        window.addEventListener('online',()=>{
            try{ client && client.stopClient(); }catch{}
            const s=load();
            if(s){
                client=createClient(s.hs,s.accessToken,s.userId,s.deviceId);
                startClient();
            }
        });
    }
    
    // Expose mount function globally
    window.awmxMount = mount;
    
    document.addEventListener('DOMContentLoaded',()=>{
        qa('.awmx-matrix__root').forEach(mount);
        qa('.awmx-matrix').forEach(mount);
    });
})();

