(function () {
  const cfg = window.AWMX || {};
  const ajax = cfg.ajax;
  const nonce = cfg.nonce;

  const root = document.querySelector('.awmx-card');
  const titleEl = document.getElementById('awmx_title');
  const listenersEl = document.getElementById('awmx_listeners');
  const djEl = document.getElementById('awmx_dj');
  const artEl = document.getElementById('awmx_art');
  const histEl = document.getElementById('awmx_history');
  const warnEl = document.querySelector('.awmx-warn');

  function inferType(stream) {
    if (!stream) return '';
    const mu = (stream.meta_url || '').toLowerCase();
    const bu = (stream.url || '').toLowerCase();
    if (mu.includes('/cp/get_info.php') || bu.includes('instainternet') || bu.includes('/sonic')) {
      return 'sonicpanel';
    }
    return stream.type || '';
  }

  function applyAlbumArt(url) {
    if (!url) return;
    try {
      if (artEl) {
        artEl.src = url;
        artEl.style.display = 'block';
      }
      if (root) {
        root.classList.add('awmx-has-bg');
        root.style.backgroundImage = 'url(\"' + url.replace(/"/g, '%22') + '\")';
      }
    } catch (e) {
      console.warn('AWMX album art apply failed', e);
    }
  }

  async function tick() {
    try {
      const s = (cfg.radio && cfg.radio.streams && cfg.radio.streams[0]) || null;
      if (!s || !ajax || !nonce) return;

      const type = (s.type && s.type.trim()) ? s.type : inferType(s);
      const payload = new URLSearchParams({
        action: 'airwave_meta',
        nonce: nonce,
        type: type || '',
        base: s.url || '',
        meta_url: s.meta_url || '',
        sonic_port: s.sonic_port || ''
      });

      const resp = await fetch(ajax, {
        method: 'POST',
        credentials: 'same-origin',
        headers: { 'Content-Type': 'application/x-www-form-urlencoded' },
        body: payload.toString()
      });

      const json = await resp.json();
      if (!json || !json.success) {
        console.warn('AWMX meta fetch failed', json);
        if (warnEl) {
          warnEl.hidden = false;
          warnEl.textContent = 'Metadata unavailable.';
        }
        return;
      }
      const d = json.data || {};

      // Title & Artist
      if (titleEl) {
        const combo = (d.artist ? (d.artist + ' — ') : '') + (d.title || '');
        titleEl.textContent = combo.trim() || '—';
      }

      // Listeners
      if (listenersEl && d.listeners !== undefined && d.listeners !== null) {
        listenersEl.textContent = String(d.listeners);
      }

      // DJ
      if (djEl && d.dj) {
        djEl.textContent = d.dj;
      }

      // Album art
      if (d.albumArt) {
        applyAlbumArt(d.albumArt);
      }

      // History
      if (histEl && Array.isArray(d.history)) {
        histEl.innerHTML = d.history.slice(0, 5).map((h) => {
          const t = (h.title || '').toString();
          return '<div class=\"awmx-history__item\">' + t.replace(/</g, '&lt;') + '</div>';
        }).join('');
      }

      if (warnEl) { warnEl.hidden = true; warnEl.textContent = ''; }

    } catch (err) {
      console.error('AWMX meta exception', err);
      if (warnEl) {
        warnEl.hidden = false;
        warnEl.textContent = 'Metadata error.';
      }
    }
  }

  // Kick and poll (default 15s)
  tick();
  const interval = (cfg.radio && Number(cfg.radio.poll_interval)) ? Number(cfg.radio.poll_interval) * 1000 : 15000;
  setInterval(tick, interval);
})();
