/**
 * Airwave Audio Player Matrix - Frontend JavaScript
 */

(function($) {
    'use strict';
    
    // Global AirwaveFrontend object
    window.AirwaveFrontend = {
        settings: {},
        audioContext: null,
        analyser: null,
        dataArray: null,
        vizTimer: null,
        matrixClient: null,
        currentRoom: null,
        refreshTimer: null,
        
        init: function() {
            this.settings = airwaveAjax.settings || {};
            this.applyTheme();
            this.initRadioPlayers();
            this.initMatrixChats();
            this.initTabs();
            this.initSongHistory();
        },
        
        applyTheme: function() {
            var theme = (this.settings.display_theme || 'dark');
            var containers = document.querySelectorAll('.airwave-radio-widget, .airwave-radio-shortcode, .airwave-combined-shortcode, .airwave-song-history-widget, .airwave-song-history-shortcode, .airwave-matrix-widget, .airwave-matrix-shortcode, .airwave-radio-content, .airwave-now-playing-shortcode, .airwave-current-song-shortcode');
            var systemDark = window.matchMedia && window.matchMedia('(prefers-color-scheme: dark)').matches;
            var resolved = theme === 'system' ? (systemDark ? 'dark' : 'light') : theme;
        
            // Toggle Airwave theme classes
            containers.forEach(function(el){
                el.classList.toggle('airwave-theme-light', resolved === 'light');
                el.classList.toggle('airwave-theme-neutral', resolved === 'neutral');
            });
        
            // Toggle AWMX card theme classes
            var cards = document.querySelectorAll('.awmx-card');
            cards.forEach(function(card){
                card.classList.toggle('awmx-theme-light', resolved === 'light');
                card.classList.toggle('awmx-theme-dark', resolved === 'dark');
                card.classList.toggle('awmx-theme-neutral', resolved === 'neutral');
            });
        
            // React to system theme changes
            if (theme === 'system' && window.matchMedia) {
                var mq = window.matchMedia('(prefers-color-scheme: dark)');
                var applySystem = function(isDark){
                    var sysResolved = isDark ? 'dark' : 'light';
                    containers.forEach(function(el){
                        el.classList.toggle('airwave-theme-light', sysResolved === 'light');
                        el.classList.toggle('airwave-theme-neutral', false);
                    });
                    cards.forEach(function(card){
                        card.classList.toggle('awmx-theme-light', sysResolved === 'light');
                        card.classList.toggle('awmx-theme-dark', sysResolved === 'dark');
                        card.classList.toggle('awmx-theme-neutral', false);
                    });
                };
                applySystem(mq.matches);
                mq.addEventListener('change', function(e){ applySystem(e.matches); });
            }
        },

        initRadioPlayers: function() {
            $('.airwave-radio-widget, .airwave-radio-shortcode, .airwave-radio-content').each(function() {
                AirwaveFrontend.initRadioPlayer($(this));
            });
        },
        
        initRadioPlayer: function($container) {
            var self = this;
            var $playBtn = $container.find('.airwave-play-btn');
            var $pauseBtn = $container.find('.airwave-pause-btn');
            var $stopBtn = $container.find('.airwave-stop-btn');
            var $volumeSlider = $container.find('.airwave-volume-slider');
            var $streamSelect = $container.find('.airwave-stream-dropdown');
            var $audioPlayer = $container.find('.airwave-audio-player');
            var $songTitle = $container.find('.airwave-song-title');
            var $songArtist = $container.find('.airwave-song-artist');
            var $albumArt = $container.find('.airwave-album-art');
            var $vizCanvas = $container.find('.airwave-viz-canvas');
            
            console.log('Initializing radio player for container:', $container);
            console.log('Found elements:', {
                playBtn: $playBtn.length,
                pauseBtn: $pauseBtn.length,
                stopBtn: $stopBtn.length,
                volumeSlider: $volumeSlider.length,
                audioPlayer: $audioPlayer.length,
                streamSelect: $streamSelect.length
            });
            
            // Populate stream dropdown
            this.populateStreamDropdown($streamSelect);
            
            // Set default stream
            var defaultStream = $container.data('stream') || 0;
            if (this.settings.streams && this.settings.streams[defaultStream]) {
                $streamSelect.val(defaultStream);
                console.log('Set default stream to:', defaultStream);
            }
            
            // Play button (unified play state; pause hidden)
            $playBtn.on('click', function() {
                var streamIndex = $streamSelect.val();
                if (!streamIndex || !self.settings.streams[streamIndex]) {
                    alert('Please select a valid stream');
                    return;
                }
                var stream = self.settings.streams[streamIndex];
                var audio = $audioPlayer[0];

                // Switch to selected stream and play
                var needsReload = (audio.src !== stream.stream_url);
                if (needsReload) {
                    audio.pause();
                    audio.src = stream.stream_url;
                    audio.volume = stream.volume || 0.8;
                    audio.load();
                }
                var playNow = function() {
                    audio.play().then(function() {
                        $playBtn.hide();
                        $pauseBtn.hide(); // remain hidden for now
                        $stopBtn.prop('disabled', false);
                    }).catch(function(error) {
                        console.error('Playback error:', error);
                        alert('Failed to play stream: ' + error.message);
                    });
                };
                if (audio.readyState === 0 || needsReload) {
                    audio.addEventListener('canplay', function onCanPlay() {
                        audio.removeEventListener('canplay', onCanPlay);
                        playNow();
                    }, { once: true });
                } else {
                    playNow();
                }

                // Setup visualizer if canvas exists
                if ($vizCanvas.length) {
                    self.setupVisualizer(audio, $vizCanvas[0]);
                }

                // Fetch metadata immediately and set refresh timer
                if (stream.api_url) {
                    self.fetchStreamMetadata(stream.api_url, $songTitle, $songArtist, $albumArt);
                    var interval = parseInt(self.settings.refresh_interval || 30, 10);
                    interval = isNaN(interval) ? 30 : Math.max(10, interval);
                    if (self.refreshTimer) {
                        clearInterval(self.refreshTimer);
                        self.refreshTimer = null;
                    }
                    self.refreshTimer = setInterval(function(){
                        self.fetchStreamMetadata(stream.api_url, $songTitle, $songArtist, $albumArt);
                    }, interval * 1000);
                }
            });
            
            // Pause button
            $pauseBtn.on('click', function() {
                $audioPlayer[0].pause();
                $playBtn.show();
                $pauseBtn.hide();
                console.log('Audio paused');
            });
            
            // Stop button (also clears metadata refresh)
            $stopBtn.on('click', function() {
                var audio = $audioPlayer[0];
                audio.pause();
                audio.currentTime = 0;
                $playBtn.show();
                $pauseBtn.hide();
                $stopBtn.prop('disabled', true);
                if (self.vizTimer) {
                    clearInterval(self.vizTimer);
                    self.vizTimer = null;
                }
                if (self.refreshTimer) {
                    clearInterval(self.refreshTimer);
                    self.refreshTimer = null;
                }
            });
            
            // Volume control
            $volumeSlider.on('input', function() {
                $audioPlayer[0].volume = $(this).val();
            });
            
            // Stream selection change
            $streamSelect.on('change', function() {
                var streamIndex = $(this).val();
                if (streamIndex && self.settings.streams[streamIndex]) {
                    var stream = self.settings.streams[streamIndex];
                    $volumeSlider.val(stream.volume || 0.8);
                }
            });
        },
        
        populateStreamDropdown: function($select) {
            $select.empty();
            $select.append('<option value="">Select Stream...</option>');
            
            $.each(this.settings.streams, function(index, stream) {
                $select.append('<option value="' + index + '">' + stream.name + '</option>');
            });
        },
        
        playStream: function(stream, $audioPlayer, $playBtn, $stopBtn) {
            var audio = $audioPlayer[0];
            
            // Prevent multiple simultaneous play requests
            if (audio.readyState === 0 && audio.src !== stream.stream_url) {
                // Set audio source and wait for it to load
                audio.src = stream.stream_url;
                audio.volume = stream.volume || 0.8;
                audio.load(); // Load the stream
                
                // Wait for the audio to be ready to play
                audio.addEventListener('canplay', function onCanPlay() {
                    audio.removeEventListener('canplay', onCanPlay);
                    audio.play().then(function() {
                        $playBtn.text('Playing...').prop('disabled', true);
                        $stopBtn.prop('disabled', false);
                    }).catch(function(error) {
                        console.error('Playback error:', error);
                        alert('Failed to play stream: ' + error.message);
                    });
                }, { once: true });
                
                // Handle load errors
                audio.addEventListener('error', function onError() {
                    audio.removeEventListener('error', onError);
                    console.error('Audio load error');
                    alert('Failed to load stream');
                }, { once: true });
                
            } else if (audio.src === stream.stream_url) {
                // Same stream, just resume playback
                audio.play().then(function() {
                    $playBtn.text('Playing...').prop('disabled', true);
                    $stopBtn.prop('disabled', false);
                }).catch(function(error) {
                    console.error('Playback error:', error);
                    alert('Failed to play stream: ' + error.message);
                });
            } else {
                // Different stream, need to reload
                audio.pause();
            audio.src = stream.stream_url;
            audio.volume = stream.volume || 0.8;
                audio.load(); // Load the stream
            
                audio.addEventListener('canplay', function onCanPlay() {
                    audio.removeEventListener('canplay', onCanPlay);
            audio.play().then(function() {
                $playBtn.text('Playing...').prop('disabled', true);
                $stopBtn.prop('disabled', false);
            }).catch(function(error) {
                console.error('Playback error:', error);
                alert('Failed to play stream: ' + error.message);
            });
                }, { once: true });
            }
        },
        
        stopStream: function($audioPlayer, $playBtn, $stopBtn) {
            var audio = $audioPlayer[0];
            audio.pause();
            audio.src = '';
            
            $playBtn.text('Play').prop('disabled', false);
            $stopBtn.prop('disabled', true);
        },
        
        setupVisualizer: function(audioElement, canvas) {
            if (this.audioContext) {
                this.audioContext.close();
            }
            
            this.audioContext = new (window.AudioContext || window.webkitAudioContext)();
            this.analyser = this.audioContext.createAnalyser();
            this.analyser.fftSize = 256;
            this.dataArray = new Uint8Array(this.analyser.frequencyBinCount);
            
            var source = this.audioContext.createMediaElementSource(audioElement);
            source.connect(this.analyser);
            this.analyser.connect(this.audioContext.destination);
            
            var ctx = canvas.getContext('2d');
            var self = this;
            
            function draw() {
                self.analyser.getByteFrequencyData(self.dataArray);
                
                ctx.fillStyle = '#1a1a1a';
                ctx.fillRect(0, 0, canvas.width, canvas.height);
                
                var barWidth = (canvas.width / self.dataArray.length) * 2.5;
                var barHeight;
                var x = 0;
                
                for (var i = 0; i < self.dataArray.length; i++) {
                    barHeight = (self.dataArray[i] / 255) * canvas.height;
                    
                    var gradient = ctx.createLinearGradient(0, canvas.height - barHeight, 0, canvas.height);
                    gradient.addColorStop(0, '#4a9eff');
                    gradient.addColorStop(1, '#1e3a8a');
                    
                    ctx.fillStyle = gradient;
                    ctx.fillRect(x, canvas.height - barHeight, barWidth, barHeight);
                    
                    x += barWidth + 1;
                }
                
                requestAnimationFrame(draw);
            }
            
            draw();
        },
        
        fetchStreamMetadata: function(apiUrl, $title, $artist, $art) {
            $.ajax({
                url: airwaveAjax.ajaxUrl,
                type: 'POST',
                data: {
                    action: 'airwave_get_stream_info',
                    nonce: airwaveAjax.nonce,
                    api_url: apiUrl
                },
                success: function(response) {
                    if (response.success) {
                        var data = response.data;
                        $title.text(data.title || '-');
                        $artist.text(data.artist || data.dj || '-');
                        
                        if (data.art) {
                            $art.attr('src', data.art).show();
                        } else {
                            $art.hide();
                        }
                    }
                },
                error: function(xhr, status, error) {
                    console.error('Failed to fetch stream metadata:', error);
                }
            });
        },
        
        initMatrixChats: function() {
            // Mount new AWMX Matrix wrappers if present
            if (typeof window.awmxMount === 'function') {
                try {
                    document.querySelectorAll('.awmx-matrix__root, .awmx-matrix').forEach(function(el){
                        window.awmxMount(el);
                    });
                } catch (e) {
                    console.error('AWMX mount failed:', e);
                }
            }
            
            $('.airwave-matrix-widget, .airwave-matrix-shortcode, .airwave-matrix-content, .airwave-chat-room-shortcode').each(function() {
                AirwaveFrontend.initMatrixChat($(this));
            });
        },
        
        initMatrixChat: function($container) {
            var self = this;
            var $status = $container.find('.airwave-matrix-status');
            var $roomList = $container.find('.airwave-room-list');
            var $roomInput = $container.find('.airwave-room-input');
            var $roomSelect = $container.find('.airwave-room-select');
            var $joinBtn = $container.find('.airwave-join-btn');
            var $currentRoom = $container.find('.airwave-room-name');
            var $messagesList = $container.find('.airwave-messages-list');
            var $messageField = $container.find('.airwave-message-field');
            var $sendBtn = $container.find('.airwave-send-btn');
            
            console.log('Initializing Matrix chat for container:', $container);
            console.log('Matrix settings:', this.settings.matrix_settings);
            
            // Check if logged in from data attribute or settings
            var isLoggedIn = $container.data('logged-in') === 'true' || (this.settings.matrix_settings && this.settings.matrix_settings.loggedIn);
            console.log('Matrix logged in:', isLoggedIn);
            
            // Initialize Matrix client if logged in
            if (isLoggedIn) {
                this.initMatrixClient($container);
            } else {
                console.log('Matrix not logged in, showing login form');
                this.initMatrixLogin($container);
            }
            
            // Join room button
            $joinBtn.on('click', function() {
                var selected = $roomSelect.length ? $roomSelect.val() : '';
                var roomId = '';
                if (selected && selected !== 'manual') {
                    roomId = selected.trim();
                } else {
                    roomId = $roomInput.val().trim();
                }
                if (!roomId) {
                    alert('Please select or enter a room ID');
                    return;
                }
                self.joinRoom(roomId, $container);
            });
            
            // Toggle manual input visibility
            $roomSelect.on('change', function() {
                var val = $(this).val();
                if (val && val !== 'manual') {
                    $roomInput.closest('.airwave-join-room').find('.airwave-room-input').hide();
                } else {
                    $roomInput.closest('.airwave-join-room').find('.airwave-room-input').show().focus();
                }
            });
            // Initialize visibility state
            $roomSelect.trigger('change');
            
            // Send message button
            $sendBtn.on('click', function() {
                var message = $messageField.val().trim();
                if (!message) {
                    return;
                }
                
                self.sendMessage(message, $container);
                $messageField.val('');
            });
            
            // Send message with Enter key
            $messageField.on('keypress', function(e) {
                if (e.which === 13) {
                    $sendBtn.click();
                }
            });
        },
        
        initMatrixClient: function($container) {
            var self = this;
            var $status = $container.find('.airwave-matrix-status');
            
            console.log('Initializing Matrix client...');
            
            // Get settings from localized script
            var settings = typeof airwaveAjax !== 'undefined' ? airwaveAjax.settings : {};
            
            if (!settings.matrix_settings || !settings.matrix_settings.homeserver) {
                console.log('Matrix settings not available');
                $status.text('Matrix settings not configured').addClass('error').show();
                return;
            }
            
            console.log('Matrix settings:', settings.matrix_settings);
            
            try {
                // Create Matrix client with real SDK
                this.matrixClient = window.matrixcs.createClient({
                    baseUrl: settings.matrix_settings.homeserver,
                    userId: settings.matrix_settings.userId,
                    accessToken: settings.matrix_settings.access_token
                });
                
                console.log('Matrix client created successfully');
                
                // Set up event handlers
                this.matrixClient.on('sync', function(state, prevState, data) {
                    console.log('Matrix sync state:', state);
                    if (state === 'SYNCING') {
                        console.log('Matrix connected successfully');
                        $status.text('Connected to Matrix').addClass('success').show();
                    } else if (state === 'ERROR') {
                        console.error('Matrix connection error');
                        $status.text('Matrix connection error').addClass('error').show();
                    }
                });
                
                this.matrixClient.on('Room.timeline', function(event, room, toStartOfTimeline, removed, data) {
                    if (event.getType() === 'm.room.message') {
                        var content = event.getContent();
                        if (content.msgtype === 'm.text') {
                            var message = {
                                sender: event.getSender(),
                                body: content.body,
                                time: new Date(event.getTs())
                            };
                            self.addMessage(message, $container);
                        }
                    }
                });
                
                // Start the client
                this.matrixClient.startClient().then(function() {
                    console.log('Matrix client started successfully');
                    
                    // Load rooms
                    self.loadMatrixRooms($container);
                    
                    // Auto-join room if specified
                    var roomId = $container.data('room');
                    if (roomId) {
                        self.joinRoom(roomId, $container);
                    }
                }).catch(function(error) {
                    console.error('Failed to start Matrix client:', error);
                    $status.text('Failed to start Matrix client: ' + error.message).addClass('error').show();
                });
                
            } catch (error) {
                console.error('Matrix client initialization error:', error);
                $status.text('Matrix client initialization error: ' + error.message).addClass('error').show();
            }
        },
        
        initMatrixLogin: function($container) {
            var self = this;
            var $loginForm = $container.find('.airwave-login-form');
            var $homeserverSelect = $container.find('.airwave-homeserver-select');
            var $customHomeserver = $container.find('.airwave-custom-homeserver');
            var $usernameInput = $container.find('.airwave-username-input');
            var $passwordInput = $container.find('.airwave-password-input');
            var $displayNameInput = $container.find('.airwave-display-name-input');
            var $loginBtn = $container.find('.airwave-login-btn');
            var $logoutBtn = $container.find('.airwave-logout-btn');
            var $loginStatus = $container.find('.airwave-login-status');
            
            console.log('Initializing Matrix login form');
            
            // Homeserver selection
            $homeserverSelect.on('change', function() {
                if ($(this).val() === 'custom') {
                    $customHomeserver.show();
                } else {
                    $customHomeserver.hide();
                }
            });
            
            // Login button
            $loginBtn.on('click', function() {
                var homeserver = $homeserverSelect.val();
                if (homeserver === 'custom') {
                    homeserver = $customHomeserver.val();
                }
                
                var username = $usernameInput.val().trim();
                var password = $passwordInput.val();
                var displayName = $displayNameInput.val().trim();
                
                if (!homeserver || !username || !password) {
                    self.showLoginStatus('Please fill in all required fields', 'error', $loginStatus);
                    return;
                }
                
                self.performMatrixLogin(homeserver, username, password, displayName, $container);
            });
            
            // Logout button
            $logoutBtn.on('click', function() {
                self.performMatrixLogout($container);
            });
        },
        
        performMatrixLogin: function(homeserver, username, password, displayName, $container) {
            var self = this;
            var $loginBtn = $container.find('.airwave-login-btn');
            var $loginStatus = $container.find('.airwave-login-status');
            var $loginForm = $container.find('.airwave-login-form');
            var $matrixContent = $container.find('.airwave-matrix-content');
            
            $loginBtn.prop('disabled', true).text('Logging in...');
            self.showLoginStatus('Logging in to Matrix...', 'info', $loginStatus);
            
            try {
                // Create Matrix client
                this.matrixClient = window.matrixcs.createClient({
                    baseUrl: homeserver
                });
                
                // Perform login
                this.matrixClient.login('m.login.password', {
                    user: username,
                    password: password
                }).then(function(response) {
                    console.log('Matrix login successful:', response);
                    
                    // Note: Display name setting is not supported in this Matrix SDK version
                    // The display name will be set during the login process
                    
                    // Update UI to show logged in state
                    $loginForm.hide();
                    
                    // Show the chat interface elements
                    $container.find('.airwave-room-selector').show();
                    $container.find('.airwave-chat-container').show();
                    
                    // Store login state in settings
                    self.settings.matrix_settings.loggedIn = true;
                    self.settings.matrix_settings.userId = response.user_id;
                    self.settings.matrix_settings.access_token = response.access_token;
                    
                    // Show success message
                    self.showLoginStatus('Successfully logged in!', 'success', $loginStatus);
                    
                    // Initialize chat interface
                    self.initMatrixClient($container);
                    
                }).catch(function(error) {
                    console.error('Matrix login error:', error);
                    self.showLoginStatus('Login failed: ' + (error.message || error), 'error', $loginStatus);
                }).finally(function() {
                    $loginBtn.prop('disabled', false).text('Login to Chat');
                });
                
            } catch (error) {
                console.error('Matrix login error:', error);
                self.showLoginStatus('Login error: ' + error.message, 'error', $loginStatus);
                $loginBtn.prop('disabled', false).text('Login to Chat');
            }
        },
        
        performMatrixLogout: function($container) {
            var $loginForm = $container.find('.airwave-login-form');
            var $matrixContent = $container.find('.airwave-matrix-content');
            var $loginStatus = $container.find('.airwave-login-status');
            
            if (this.matrixClient) {
                this.matrixClient.logout();
            }
            
            this.matrixClient = null;
            this.currentRoom = null;
            
            $matrixContent.hide();
            $loginForm.show();
            
            // Clear form
            $container.find('.airwave-username-input').val('');
            $container.find('.airwave-password-input').val('');
            $container.find('.airwave-display-name-input').val('');
            
            this.showLoginStatus('Logged out successfully', 'success', $loginStatus);
        },
        
        showLoginStatus: function(message, type, $status) {
            $status.removeClass('success error info warning').addClass(type).text(message).show();
            
            if (type === 'success' || type === 'error') {
                setTimeout(function() {
                    $status.hide();
                }, 5000);
            }
        },
        
        addSampleRooms: function($container) {
            var $roomList = $container.find('.airwave-room-list');
            var sampleRooms = [
                { id: '#matrix:matrix.org', name: 'Official Matrix Room' },
                { id: '#general:matrix.org', name: 'General Discussion' },
                { id: '#help:matrix.org', name: 'Help & Support' },
                { id: '#airwavesupport:matrix.org', name: 'Airwave Support' }
            ];
            
            $.each(sampleRooms, function(index, room) {
                var $roomItem = $('<div class="airwave-room-item" data-room-id="' + room.id + '">' + room.name + '</div>');
                $roomItem.on('click', function() {
                    AirwaveFrontend.joinRoom(room.id, $container);
                });
                $roomList.append($roomItem);
            });
        },
        
        joinRoom: function(roomId, $container) {
            var $currentRoom = $container.find('.airwave-room-name');
            var $roomItems = $container.find('.airwave-room-item');
            
            // Update current room display
            var roomName = roomId.replace('#', '').replace(':matrix.org', '');
            $currentRoom.text('Current room: ' + roomName);
            
            // Update room selection
            $roomItems.removeClass('active');
            $roomItems.filter('[data-room-id="' + roomId + '"]').addClass('active');
            
            // Load messages for the room
            this.loadRoomMessages(roomId, $container);
            
            this.currentRoom = roomId;
        },
        
        loadRoomMessages: function(roomId, $container) {
            var $messagesList = $container.find('.airwave-messages-list');
            
            // Clear existing messages
            $messagesList.empty();
            
            // Add loading message
            $messagesList.append('<div class="airwave-message"><div class="airwave-message-body">Loading messages...</div></div>');
            
            // Simulate loading messages
            setTimeout(function() {
                $messagesList.empty();
                
                // Add sample messages
                var sampleMessages = [
                    { sender: 'testuser', body: 'Hello from Matrix!', time: new Date() },
                    { sender: 'admin', body: 'Welcome to the room!', time: new Date() }
                ];
                
                $.each(sampleMessages, function(index, message) {
                    AirwaveFrontend.addMessage(message, $container);
                });
            }, 1000);
        },
        
        sendMessage: function(message, $container) {
            if (!this.currentRoom) {
                alert('Please join a room first');
                return;
            }
            
            // Add message to UI immediately
            var messageObj = {
                sender: 'You',
                body: message,
                time: new Date()
            };
            
            this.addMessage(messageObj, $container);
            
            // In a real implementation, this would send the message via Matrix API
            console.log('Sending message to', this.currentRoom, ':', message);
        },
        
        addMessage: function(message, $container) {
            var $messagesList = $container.find('.airwave-messages-list');
            var timeStr = message.time.toLocaleTimeString([], {hour: '2-digit', minute:'2-digit'});
            
            var $messageDiv = $('<div class="airwave-message">' +
                '<div class="airwave-message-header">' +
                    '<div class="airwave-message-sender">' + message.sender + '</div>' +
                    '<div class="airwave-message-time">' + timeStr + '</div>' +
                '</div>' +
                '<div class="airwave-message-body">' + message.body + '</div>' +
            '</div>');
            
            $messagesList.append($messageDiv);
            
            // Scroll to bottom
            var $messagesContainer = $container.find('.airwave-messages-container');
            $messagesContainer.scrollTop($messagesContainer[0].scrollHeight);
        },
        
        initTabs: function() {
            $('.airwave-tabs').each(function() {
                AirwaveFrontend.initTabContainer($(this));
            });
        },
        
        initTabContainer: function($tabs) {
            var $tabBtns = $tabs.find('.airwave-tab-btn');
            var $tabPanels = $tabs.find('.airwave-tab-panel');
            
            $tabBtns.on('click', function() {
                var targetTab = $(this).data('tab');
                
                // Update button states
                $tabBtns.removeClass('active');
                $(this).addClass('active');
                
                // Update panel states
                $tabPanels.removeClass('active');
                $tabPanels.filter('#' + targetTab).addClass('active');
            });
        },
        
        initSongHistory: function() {
            $('.airwave-song-history-widget, .airwave-song-history-shortcode').each(function() {
                AirwaveFrontend.initSongHistoryWidget($(this));
            });
            
            // Initialize Now Playing shortcode (without Recent Songs)
            $('.airwave-now-playing-shortcode, .airwave-current-song-shortcode').each(function() {
                AirwaveFrontend.initNowPlayingWidget($(this));
            });
        },
        
        initSongHistoryWidget: function($container) {
            var streamIndex = $container.data('stream') || 0;
            var limit = $container.data('limit') || 10;
            
            if (this.settings.streams[streamIndex]) {
                var stream = this.settings.streams[streamIndex];
                
                // Update current song
                var $songTitle = $container.find('.airwave-song-title');
                var $songArtist = $container.find('.airwave-song-artist');
                var $albumArt = $container.find('.airwave-album-art');
                
                if (stream.api_url) {
                    this.fetchStreamMetadata(stream.api_url, $songTitle, $songArtist, $albumArt);
                }
                
                // Load song history
                this.loadSongHistory(stream.api_url, $container, limit);
            }
        },
        
        loadSongHistory: function(apiUrl, $container, limit) {
            var $historyList = $container.find('.airwave-history-list');
            
            if (!apiUrl) {
                $historyList.html('<div class="airwave-history-item">No history available</div>');
                return;
            }
            
            $.ajax({
                url: airwaveAjax.ajaxUrl,
                type: 'POST',
                data: {
                    action: 'airwave_get_stream_info',
                    nonce: airwaveAjax.nonce,
                    api_url: apiUrl
                },
                success: function(response) {
                    if (response.success && response.data.history) {
                        var history = response.data.history.slice(0, limit);
                        $historyList.empty();
                        
                        $.each(history, function(index, song) {
                            $historyList.append('<div class="airwave-history-item">' + song + '</div>');
                        });
                    } else {
                        $historyList.html('<div class="airwave-history-item">No history available</div>');
                    }
                },
                error: function() {
                    $historyList.html('<div class="airwave-history-item">Failed to load history</div>');
                }
            });
        },
        
        initNowPlayingWidget: function($container) {
            var streamIndex = $container.data('stream') || 0;
            var self = this;
            
            if (this.settings.streams && this.settings.streams[streamIndex]) {
                var stream = this.settings.streams[streamIndex];
                
                // Update current song
                var $songTitle = $container.find('.airwave-song-title');
                var $songArtist = $container.find('.airwave-song-artist');
                var $albumArt = $container.find('.airwave-album-art');
                var $listenersValue = $container.find('.airwave-listeners-value');
                var $djValue = $container.find('.airwave-dj-value');
                
                // Fetch metadata using the AJAX endpoint
                if (stream.api_url || stream.meta_url || stream.url) {
                    // Use the metadata AJAX endpoint
                    $.ajax({
                        url: airwaveAjax.ajaxUrl,
                        type: 'POST',
                        data: {
                            action: 'airwave_meta',
                            nonce: airwaveAjax.nonce,
                            type: stream.type || 'sonicpanel',
                            base: stream.url || '',
                            meta_url: stream.meta_url || stream.api_url || '',
                            sonic_port: stream.sonic_port || ''
                        },
                        success: function(response) {
                            if (response.success && response.data) {
                                var data = response.data;
                                
                                // Update title and artist
                                if (data.title) {
                                    $songTitle.text(data.title);
                                }
                                if (data.artist) {
                                    $songArtist.text(data.artist);
                                }
                                
                                // Update album art
                                if (data.albumArt && $albumArt.length) {
                                    $albumArt.attr('src', data.albumArt);
                                    $albumArt.css('display', 'block');
                                }
                                
                                // Update listeners
                                if ($listenersValue.length && data.listeners !== undefined && data.listeners !== null) {
                                    $listenersValue.text(data.listeners);
                                }
                                
                                // Update DJ
                                if ($djValue.length && data.dj) {
                                    $djValue.text(data.dj);
                                }
                            }
                        },
                        error: function() {
                            console.error('Failed to fetch metadata for Now Playing widget');
                        }
                    });
                    
                    // Set up polling every 15 seconds
                    var pollInterval = setInterval(function() {
                        if (!$container.is(':visible') || !$container.length) {
                            clearInterval(pollInterval);
                            return;
                        }
                        
                        $.ajax({
                            url: airwaveAjax.ajaxUrl,
                            type: 'POST',
                            data: {
                                action: 'airwave_meta',
                                nonce: airwaveAjax.nonce,
                                type: stream.type || 'sonicpanel',
                                base: stream.url || '',
                                meta_url: stream.meta_url || stream.api_url || '',
                                sonic_port: stream.sonic_port || ''
                            },
                            success: function(response) {
                                if (response.success && response.data) {
                                    var data = response.data;
                                    
                                    if (data.title) {
                                        $songTitle.text(data.title);
                                    }
                                    if (data.artist) {
                                        $songArtist.text(data.artist);
                                    }
                                    
                                    if (data.albumArt && $albumArt.length) {
                                        $albumArt.attr('src', data.albumArt);
                                        $albumArt.css('display', 'block');
                                    }
                                    
                                    if ($listenersValue.length && data.listeners !== undefined && data.listeners !== null) {
                                        $listenersValue.text(data.listeners);
                                    }
                                    
                                    if ($djValue.length && data.dj) {
                                        $djValue.text(data.dj);
                                    }
                                }
                            }
                        });
                    }, 15000);
                }
            }
        }
    };
    
    // Initialize when document is ready
    $(document).ready(function() {
        if (typeof airwaveAjax !== 'undefined') {
            AirwaveFrontend.init();
        }
    });
    
})(jQuery);
