/**
 * Simplified Matrix SDK for Airwave Audio Player Matrix
 * This is a basic implementation for demonstration purposes
 */

(function() {
    'use strict';
    
    // Simple Matrix client implementation
    window.MatrixClient = function(options) {
        this.baseUrl = options.baseUrl;
        this.accessToken = options.accessToken;
        this.userId = options.userId;
        this.deviceId = options.deviceId;
        this.rooms = new Map();
        this.eventHandlers = new Map();
    };
    
    MatrixClient.prototype.login = function(loginType, credentials) {
        return new Promise((resolve, reject) => {
            // Simulate login
            setTimeout(() => {
                if (credentials.user && credentials.password) {
                    resolve({
                        access_token: 'mock_access_token',
                        user_id: credentials.user,
                        device_id: 'mock_device_id'
                    });
                } else {
                    reject(new Error('Invalid credentials'));
                }
            }, 1000);
        });
    };
    
    MatrixClient.prototype.joinRoom = function(roomId) {
        return new Promise((resolve, reject) => {
            // Simulate joining room
            setTimeout(() => {
                if (roomId.startsWith('#')) {
                    this.rooms.set(roomId, {
                        roomId: roomId,
                        name: roomId.replace('#', '').replace(':matrix.org', ''),
                        timeline: { events: [] }
                    });
                    resolve({ room_id: roomId });
                } else {
                    reject(new Error('Invalid room ID'));
                }
            }, 500);
        });
    };
    
    MatrixClient.prototype.sendTextMessage = function(roomId, text) {
        return new Promise((resolve, reject) => {
            // Simulate sending message
            setTimeout(() => {
                var room = this.rooms.get(roomId);
                if (room) {
                    var event = {
                        event_id: 'mock_' + Date.now(),
                        type: 'm.room.message',
                        content: {
                            body: text,
                            msgtype: 'm.text'
                        },
                        sender: this.userId,
                        ts: Date.now()
                    };
                    
                    room.timeline.events.push(event);
                    
                    // Trigger event handler
                    this.emit('Room.timeline', event, room);
                    
                    resolve({ event_id: event.event_id });
                } else {
                    reject(new Error('Room not found'));
                }
            }, 300);
        });
    };
    
    MatrixClient.prototype.getRooms = function() {
        return Array.from(this.rooms.values());
    };
    
    MatrixClient.prototype.getRoom = function(roomId) {
        return this.rooms.get(roomId);
    };
    
    MatrixClient.prototype.on = function(event, handler) {
        if (!this.eventHandlers.has(event)) {
            this.eventHandlers.set(event, []);
        }
        this.eventHandlers.get(event).push(handler);
    };
    
    MatrixClient.prototype.once = function(event, handler) {
        var self = this;
        var onceHandler = function() {
            handler.apply(this, arguments);
            self.off(event, onceHandler);
        };
        this.on(event, onceHandler);
    };
    
    MatrixClient.prototype.off = function(event, handler) {
        if (this.eventHandlers.has(event)) {
            var handlers = this.eventHandlers.get(event);
            var index = handlers.indexOf(handler);
            if (index > -1) {
                handlers.splice(index, 1);
            }
        }
    };
    
    MatrixClient.prototype.emit = function(event) {
        if (this.eventHandlers.has(event)) {
            var handlers = this.eventHandlers.get(event);
            var args = Array.prototype.slice.call(arguments, 1);
            handlers.forEach(function(handler) {
                handler.apply(null, args);
            });
        }
    };
    
    MatrixClient.prototype.startClient = function(options) {
        return new Promise((resolve) => {
            // Simulate starting client
            setTimeout(() => {
                this.emit('sync', 'SYNCING');
                resolve();
            }, 1000);
        });
    };
    
    MatrixClient.prototype.stopClient = function() {
        // Simulate stopping client
        console.log('Matrix client stopped');
    };
    
    MatrixClient.prototype.performSync = function(limit) {
        return new Promise((resolve) => {
            // Simulate sync
            setTimeout(() => {
                this.emit('sync', 'SYNCING');
                resolve();
            }, 500);
        });
    };
    
    MatrixClient.prototype.resetSyncToken = function() {
        // Reset sync token
        console.log('Sync token reset');
    };
    
    // Create Matrix SDK namespace
    window.matrixcs = {
        createClient: function(options) {
            return new MatrixClient(options);
        }
    };
    
})();
