<?php
/**
 * Admin interface for Airwave Audio Player Matrix
 */

if (!defined('ABSPATH')) {
    exit;
}

class AirwaveAdmin {
    
    public function __construct() {
        add_action('admin_menu', array($this, 'add_admin_menu'));
        add_action('admin_init', array($this, 'register_settings'));
        add_action('admin_enqueue_scripts', array($this, 'enqueue_admin_scripts'));
        add_action('admin_enqueue_scripts', array($this, 'enqueue_status_scripts'));
        add_action('wp_enqueue_scripts', array($this, 'enqueue_status_scripts'));
        add_action('admin_bar_menu', array($this, 'add_admin_bar_items'), 100);
        
        // AJAX handlers
        add_action('wp_ajax_airwave_save_settings', array($this, 'save_settings'));
        add_action('wp_ajax_airwave_test_stream', array($this, 'test_stream'));
        add_action('wp_ajax_airwave_test_all_streams', array($this, 'test_all_streams'));
        add_action('wp_ajax_airwave_test_matrix', array($this, 'test_matrix'));
        add_action('wp_ajax_airwave_test_sonic', array($this, 'test_sonic'));
        add_action('wp_ajax_airwave_get_status', array($this, 'get_status'));
        add_action('wp_ajax_airwave_matrix_login', array($this, 'matrix_login'));
        add_action('wp_ajax_airwave_matrix_logout', array($this, 'matrix_logout'));
        add_action('wp_ajax_airwave_matrix_admin_login', array($this, 'matrix_admin_login'));
        add_action('wp_ajax_airwave_matrix_admin_logout', array($this, 'matrix_admin_logout'));
        add_action('wp_ajax_airwave_matrix_admin_status', array($this, 'matrix_admin_status'));
        add_action('wp_ajax_airwave_apply_test_credentials', array($this, 'apply_test_credentials'));
        
        // Debug system AJAX handlers
        add_action('wp_ajax_airwave_test_system', array($this, 'test_system'));
        add_action('wp_ajax_airwave_get_debug_log', array($this, 'get_debug_log'));
        add_action('wp_ajax_airwave_clear_debug_log', array($this, 'clear_debug_log'));
        add_action('wp_ajax_airwave_download_debug_log', array($this, 'download_debug_log'));
        
        // Force version refresh notice
        add_action('admin_notices', array($this, 'version_refresh_notice'));
        
        // Rate limiting for AJAX endpoints
        add_action('wp_ajax_airwave_test_stream', array($this, 'rate_limit_check'), 1);
        add_action('wp_ajax_airwave_test_matrix', array($this, 'rate_limit_check'), 1);
        add_action('wp_ajax_airwave_test_sonic', array($this, 'rate_limit_check'), 1);
        add_action('wp_ajax_airwave_test_all_streams', array($this, 'rate_limit_check'), 1);
        add_action('wp_ajax_airwave_test_system', array($this, 'rate_limit_check'), 1);

        // Redirect malformed admin URLs carrying Matrix credentials to Matrix page
        add_action('admin_init', array($this, 'redirect_matrix_query'));

        // Suppress non-Airwave admin notices on Airwave admin screens
        add_action('current_screen', array($this, 'suppress_external_admin_notices'));
    }
    
    public function rate_limit_check() {
        $user_id = get_current_user_id();
        $action = $_POST['action'] ?? 'unknown';
        $key = 'airwave_rate_limit_' . $user_id . '_' . $action;
        $requests = get_transient($key) ?: 0;
        
        // Allow 10 requests per minute for testing endpoints
        if ($requests >= 10) {
            wp_send_json_error('Rate limit exceeded. Please wait before trying again.');
        }
        
        set_transient($key, $requests + 1, 60);
    }

    // Redirect admin.php requests with Matrix credential query params to Matrix page
    public function redirect_matrix_query() {
        if (!is_admin()) {
            return;
        }
        $page = isset($_GET['page']) ? sanitize_text_field($_GET['page']) : '';
        $hasCredParams = (isset($_GET['username']) || isset($_GET['password']));
        if ($hasCredParams && $page !== 'airwave-matrix') {
            wp_safe_redirect(admin_url('admin.php?page=airwave-matrix'));
            exit;
        }
    }
    
    // Remove unrelated admin notices on Airwave plugin pages
    public function suppress_external_admin_notices($screen) {
        if (!$screen) {
            return;
        }
        if (strpos($screen->id, 'airwave') !== false) {
            remove_all_actions('admin_notices');
            remove_all_actions('all_admin_notices');
            // Re-add our own minimal notice when appropriate
            add_action('admin_notices', array($this, 'version_refresh_notice'));
        }
    }

    public function version_refresh_notice() {
        // Only show on plugin pages
        $screen = get_current_screen();
        if (!$screen || strpos($screen->id, 'airwave') === false) {
            return;
        }
        
        // Check if this is the first load after version update
        $last_version_check = get_option('airwave_last_version_check', '0.0.45');
        if ($last_version_check !== AIRWAVE_PLUGIN_VERSION) {
            update_option('airwave_last_version_check', AIRWAVE_PLUGIN_VERSION);
            ?>
            <div class="notice notice-success is-dismissible">
                <p><strong>Airwave Audio Player Matrix</strong> has been updated to version <strong><?php echo AIRWAVE_PLUGIN_VERSION; ?></strong>! 
                <a href="<?php echo admin_url('admin.php?page=airwave-settings&tab=matrix'); ?>">View new features</a></p>
            </div>
            <?php
        }
    }
    
    public function add_admin_menu() {
        // Add main menu page
        add_menu_page(
            'Airwave Audio Player Matrix',
            'Airwave Matrix',
            'manage_options',
            'airwave-dashboard',
            array($this, 'dashboard_page'),
            'dashicons-controls-play',
            30
        );
        
        // Add Settings submenu page
        add_submenu_page(
            'airwave-dashboard',
            'Airwave Matrix Settings',
            'Settings',
            'manage_options',
            'airwave-settings',
            array($this, 'settings_page')
        );
        
        // Add Matrix Chat submenu page
        add_submenu_page(
            'airwave-dashboard',
            'Airwave Matrix Chat',
            'Matrix Chat',
            'manage_options',
            'airwave-matrix',
            array($this, 'matrix_page')
        );
        
        // Add Shortcodes submenu page
        add_submenu_page(
            'airwave-dashboard',
            'Airwave Matrix Shortcodes',
            'Shortcodes',
            'manage_options',
            'airwave-shortcodes',
            array($this, 'shortcodes_page')
        );
        
        // Add Testing submenu page
        add_submenu_page(
            'airwave-dashboard',
            'Airwave Matrix Testing',
            'Testing',
            'manage_options',
            'airwave-testing',
            array($this, 'testing_page')
        );
        
        // Add Documentation submenu page
        add_submenu_page(
            'airwave-dashboard',
            'Airwave Matrix Documentation',
            'Documentation',
            'manage_options',
            'airwave-documentation',
            array($this, 'documentation_page')
        );
        
        // Add Clear Cache submenu page
        add_submenu_page(
            'airwave-dashboard',
            'Clear Cache',
            'Clear Cache',
            'manage_options',
            'airwave-clear-cache',
            array($this, 'clear_cache_page')
        );
    }
    
    public function register_settings() {
        register_setting(
            'airwave_settings',
            'airwave_settings',
            array(
                'type' => 'array',
                'sanitize_callback' => array($this, 'sanitize_settings'),
                'default' => array()
            )
        );
    }
    
    public function sanitize_settings($input) {
        // Verify nonce if present
        if (isset($_POST['airwave_settings_nonce']) && !wp_verify_nonce($_POST['airwave_settings_nonce'], 'airwave_settings')) {
            return get_option('airwave_settings', array());
        }
        
        // If input is not an array, return existing settings
        if (!is_array($input)) {
            return get_option('airwave_settings', array());
        }
        
        $output = array();
        
        // Radio settings
        $mode = isset($input['radio_mode']) ? $input['radio_mode'] : 'single';
        $output['radio_mode'] = in_array($mode, ['single', 'rotate'], true) ? $mode : 'single';
        $output['rotate_interval'] = max(10, intval($input['rotate_interval'] ?? 60));
        $output['autoplay'] = !empty($input['autoplay']) ? 1 : 0;
        
        // Stream settings (1-5)
        for ($i = 1; $i <= 5; $i++) {
            $prefix = "stream_$i";
            $output["{$prefix}_name"] = sanitize_text_field($input["{$prefix}_name"] ?? '');
            $output["{$prefix}_type"] = in_array($input["{$prefix}_type"] ?? '', 
                ['shoutcast', 'icecast', 'sonicpanel'], true) ? $input["{$prefix}_type"] : '';
            $output["{$prefix}_url"] = esc_url_raw($input["{$prefix}_url"] ?? '');
            $output["{$prefix}_api_url"] = esc_url_raw($input["{$prefix}_api_url"] ?? '');
            $output["{$prefix}_meta_url"] = esc_url_raw($input["{$prefix}_meta_url"] ?? '');
            $output["{$prefix}_sonic_port"] = preg_replace('/[^0-9]/', '', $input["{$prefix}_sonic_port"] ?? '');
        }
        
        // Matrix settings
        $output['matrix_homeserver'] = esc_url_raw($input['matrix_homeserver'] ?? 'https://matrix-client.matrix.org');
        $output['matrix_custom_homeserver'] = esc_url_raw($input['matrix_custom_homeserver'] ?? '');
        $output['matrix_username'] = sanitize_text_field($input['matrix_username'] ?? '');
        $output['matrix_password'] = sanitize_text_field($input['matrix_password'] ?? '');
        $output['matrix_room'] = sanitize_text_field($input['matrix_room'] ?? '');
        $output['refresh_interval'] = max(5, intval($input['refresh_interval'] ?? 30));
        $output['stream_page_url'] = esc_url_raw($input['stream_page_url'] ?? '');
        $output['chat_page_url'] = esc_url_raw($input['chat_page_url'] ?? '');
        
        // Display settings
        $output['display_theme'] = in_array($input['display_theme'] ?? 'dark', ['dark', 'light', 'neutral', 'system'], true) ? ($input['display_theme'] ?? 'dark') : 'dark';
        $output['default_cover_image'] = absint($input['default_cover_image'] ?? 0); // Attachment ID from media library
        
        // Sonic Panel settings
        $output['sonic_enabled'] = !empty($input['sonic_enabled']) ? 1 : 0;
        $output['sonic_server_url'] = esc_url_raw($input['sonic_server_url'] ?? 'https://sonic24.instainternet.com');
        $output['sonic_port'] = preg_replace('/[^0-9]/', '', $input['sonic_port'] ?? '8416');
        $output['sonic_stream_url'] = esc_url_raw($input['sonic_stream_url'] ?? 'https://sonic24.instainternet.com/8416/stream');
        $output['sonic_json_api'] = esc_url_raw($input['sonic_json_api'] ?? 'https://sonic24.instainternet.com/cp/get_info.php?p=8416');
        $output['sonic_admin_url'] = esc_url_raw($input['sonic_admin_url'] ?? 'http://sonic24.instainternet.com:8416/admin.cgi');
        $output['sonic_username'] = sanitize_text_field($input['sonic_username'] ?? 'admin');
        $output['sonic_password'] = sanitize_text_field($input['sonic_password'] ?? '');
        $output['sonic_widget_script'] = esc_url_raw($input['sonic_widget_script'] ?? 'https://sonic24.instainternet.com/cp/widgets.js?r=684');
        
        return $output;
    }
    
    public function enqueue_admin_scripts($hook) {
        if (strpos($hook, 'airwave') !== false) {
            wp_enqueue_style(
                'airwave-admin-css',
                AIRWAVE_PLUGIN_URL . 'assets/css/admin.css',
                array(),
                AIRWAVE_PLUGIN_VERSION
            );
            wp_enqueue_style(
                'airwave-frontend-css',
                AIRWAVE_PLUGIN_URL . 'assets/css/frontend.css',
                array(),
                AIRWAVE_PLUGIN_VERSION
            );
            wp_enqueue_style(
                'airwave-awmx-css',
                AIRWAVE_PLUGIN_URL . 'assets/css/awmx.css',
                array(),
                AIRWAVE_PLUGIN_VERSION
            );
            
            // Enqueue media uploader for cover image selection
            if (strpos($hook, 'airwave-settings') !== false) {
                wp_enqueue_media();
            }
            
            wp_enqueue_script(
                'airwave-admin-js',
                AIRWAVE_PLUGIN_URL . 'assets/js/admin.js',
                array('jquery'),
                AIRWAVE_PLUGIN_VERSION,
                true
            );
            
            wp_enqueue_script(
                'airwave-status-js',
                AIRWAVE_PLUGIN_URL . 'assets/js/status.js',
                array('jquery'),
                AIRWAVE_PLUGIN_VERSION,
                true
            );
            
            // Enqueue frontend scripts for shortcode testing
            wp_enqueue_script(
                'airwave-frontend-js',
                AIRWAVE_PLUGIN_URL . 'assets/js/frontend.js',
                array('jquery'),
                AIRWAVE_PLUGIN_VERSION,
                true
            );
            
            wp_localize_script('airwave-admin-js', 'airwaveAjax', array(
                'ajaxUrl' => admin_url('admin-ajax.php'),
                'nonce' => wp_create_nonce('airwave_nonce'),
                'settings' => get_option('airwave_settings', array())
            ));
            
            wp_localize_script('airwave-status-js', 'airwaveStatus', array(
                'ajax_url' => admin_url('admin-ajax.php'),
                'nonce' => wp_create_nonce('airwave_status_nonce')
            ));
            
            // Build structured frontend settings for admin shortcode testing
            $raw_settings = get_option('airwave_settings', array());
            $streams = array();
            for ($i = 1; $i <= 5; $i++) {
                $url = $raw_settings["stream_{$i}_url"] ?? '';
                if (!empty($url)) {
                    $streams[] = array(
                        'name' => $raw_settings["stream_{$i}_name"] ?? "Stream {$i}",
                        'stream_url' => $url,
                        'api_url' => $raw_settings["stream_{$i}_api_url"] ?? ''
                    );
                }
            }
            $homeserver = $raw_settings['matrix_homeserver'] ?? '';
            if ($homeserver === 'custom') {
                $homeserver = $raw_settings['matrix_custom_homeserver'] ?? '';
            }
            $matrix_settings = array(
                'homeserver' => $homeserver,
                'room' => $raw_settings['matrix_room'] ?? ''
            );
            $frontend_settings = array(
                'streams' => $streams,
                'matrix_settings' => $matrix_settings,
                'display_theme' => $raw_settings['display_theme'] ?? 'dark',
                'refresh_interval' => intval($raw_settings['refresh_interval'] ?? 30)
            );

            // Localize frontend script with structured settings expected by frontend.js
            wp_localize_script('airwave-frontend-js', 'airwaveAjax', array(
                'ajaxUrl' => admin_url('admin-ajax.php'),
                'nonce' => wp_create_nonce('airwave_nonce'),
                'settings' => $frontend_settings
            ));
        }
    }
    
    public function enqueue_status_scripts() {
        wp_enqueue_style(
            'airwave-admin-css',
            AIRWAVE_PLUGIN_URL . 'assets/css/admin.css',
            array(),
            AIRWAVE_PLUGIN_VERSION
        );
        
        wp_enqueue_script(
            'airwave-status-js',
            AIRWAVE_PLUGIN_URL . 'assets/js/status.js',
            array('jquery'),
            AIRWAVE_PLUGIN_VERSION,
            true
        );
        
        wp_localize_script('airwave-status-js', 'airwaveStatus', array(
            'ajax_url' => admin_url('admin-ajax.php'),
            'nonce' => wp_create_nonce('airwave_status_nonce')
        ));
    }
    
    public function add_admin_bar_items($wp_admin_bar) {
        // Show admin bar items for all logged-in users
        if (!is_user_logged_in()) {
            return;
        }
        
        // Get configured page URLs
        $settings = get_option('airwave_settings', array());
        $stream_page_url = $settings['stream_page_url'] ?? home_url('/radio/');
        $chat_page_url = $settings['chat_page_url'] ?? home_url('/chat/');
        
        // Add stream status indicator
        $wp_admin_bar->add_node(array(
            'id' => 'airwave-stream-status',
            'title' => '<span class="airwave-status-indicator offline" id="airwave-stream-indicator"></span> Stream',
            'href' => $stream_page_url,
            'meta' => array(
                'class' => 'airwave-status-item',
                'title' => 'Click to go to stream page'
            )
        ));
        
        // Add Matrix status indicator
        $wp_admin_bar->add_node(array(
            'id' => 'airwave-matrix-status',
            'title' => '<span class="airwave-status-indicator offline" id="airwave-matrix-indicator"></span> Matrix',
            'href' => $chat_page_url,
            'meta' => array(
                'class' => 'airwave-status-item',
                'title' => 'Click to go to chat page'
            )
        ));
    }
    
    // Dashboard Page
    public function dashboard_page() {
        ?>
        <div class="wrap">
            <div class="airwave-header">
                <?php 
                $logo_url = AIRWAVE_PLUGIN_URL . 'assets/logo.png';
                $logo_path = AIRWAVE_PLUGIN_PATH . 'assets/logo.png';
                if (file_exists($logo_path)): ?>
                    <img src="<?php echo esc_url($logo_url); ?>" alt="Airwave Audio Player Matrix" class="airwave-logo" onerror="this.style.display='none';">
                <?php else: ?>
                    <span class="dashicons dashicons-controls-play" style="font-size: 40px; width: 40px; height: 40px; line-height: 40px; margin-right: 15px;"></span>
                <?php endif; ?>
                <h1>Airwave Audio Player Matrix</h1>
            </div>
            
            <div class="airwave-dashboard">
                <div class="airwave-dashboard-widgets">
                    <div class="airwave-widget">
                        <h3>Quick Stats</h3>
                        <p>Plugin Version: <?php echo AIRWAVE_PLUGIN_VERSION; ?></p>
                        <p>WordPress Version: <?php echo get_bloginfo('version'); ?></p>
                        <p>PHP Version: <?php echo PHP_VERSION; ?></p>
                    </div>
                    
                    <div class="airwave-widget">
                        <h3>Quick Actions</h3>
                        <p><a href="<?php echo admin_url('admin.php?page=airwave-settings'); ?>" class="button">Configure Settings</a></p>
                        <p><a href="<?php echo admin_url('admin.php?page=airwave-shortcodes'); ?>" class="button">View Shortcodes</a></p>
                        <p><a href="<?php echo admin_url('admin.php?page=airwave-testing'); ?>" class="button">Run Tests</a></p>
                    </div>
                    
                    <div class="airwave-widget">
                        <h3>Support & Contact</h3>
                        <p><strong>Author:</strong> Spun Web Technology</p>
                        <p><strong>Matrix:</strong> <a href="https://matrix.to/#/@disruptarianradio:matrix.org" target="_blank">@disruptarianradio:matrix.org</a></p>
                        <p><strong>Support Channel:</strong> <a href="https://matrix.to/#/#airwavesupport:matrix.org" target="_blank">#airwavesupport:matrix.org</a></p>
                        <p><strong>Toll Free:</strong> <a href="tel:+18882646790">+1 (888) 264‑6790</a></p>
                        <p><strong>Email:</strong> <a href="mailto:support@spunwebtechnology.com">support@spunwebtechnology.com</a></p>
                        <p><strong>Website:</strong> <a href="https://spunwebtechnology.com" target="_blank">spunwebtechnology.com</a></p>
                    </div>
                </div>
            </div>
        </div>
        <?php
    }
    
    // Settings Page with Tabs
    public function settings_page() {
        $settings = get_option('airwave_settings', array());
        $active_tab = isset($_GET['tab']) ? $_GET['tab'] : 'streams';
        ?>
        <div class="wrap">
            <div class="airwave-header">
                <?php 
                $logo_url = AIRWAVE_PLUGIN_URL . 'assets/logo.png';
                $logo_path = AIRWAVE_PLUGIN_PATH . 'assets/logo.png';
                if (file_exists($logo_path)): ?>
                    <img src="<?php echo esc_url($logo_url); ?>" alt="Airwave Audio Player Matrix" class="airwave-logo" onerror="this.style.display='none';">
                <?php else: ?>
                    <span class="dashicons dashicons-controls-play" style="font-size: 40px; width: 40px; height: 40px; line-height: 40px; margin-right: 15px;"></span>
                <?php endif; ?>
                <h1>Airwave Audio Player Matrix Settings</h1>
            </div>
            
            <h2 class="nav-tab-wrapper">
                <a href="?page=airwave-settings&tab=streams" class="nav-tab <?php echo $active_tab == 'streams' ? 'nav-tab-active' : ''; ?>">Stream Configuration</a>

                <a href="?page=airwave-settings&tab=matrix" class="nav-tab <?php echo $active_tab == 'matrix' ? 'nav-tab-active' : ''; ?>">Matrix Configuration</a>
                <a href="?page=airwave-settings&tab=display" class="nav-tab <?php echo $active_tab == 'display' ? 'nav-tab-active' : ''; ?>">Display Settings</a>
            </h2>
            
            <form method="post" action="options.php" id="airwave-settings-form">
                <?php settings_fields('airwave_settings'); ?>
                <?php wp_nonce_field('airwave_settings', 'airwave_settings_nonce'); ?>
                
                <?php if ($active_tab == 'streams'): ?>
                    <?php $this->render_streams_tab($settings); ?>
                <?php elseif ($active_tab == 'sonic'): ?>
                    <?php $this->render_sonic_tab($settings); ?>
                <?php elseif ($active_tab == 'matrix'): ?>
                    <?php $this->render_matrix_tab($settings); ?>
                <?php elseif ($active_tab == 'display'): ?>
                    <?php $this->render_display_tab($settings); ?>
                <?php endif; ?>
                
                <p class="submit">
                    <?php submit_button('Save Settings', 'primary', 'submit', false); ?>
                    <span class="airwave-save-message" id="airwave-save-message" style="display:none; margin-left: 15px;"></span>
                </p>
            </form>
        </div>
        <?php
    }
    
    private function render_streams_tab($settings) {
        ?>
        <div class="airwave-tab-content">
            <h3>Radio Stream Configuration</h3>
            <p>Configure your radio streams. Start with Stream 1, then add additional streams as needed (up to 5 total).</p>
            
            <div class="airwave-stream-config">
                <div class="airwave-inline">
                    <label for="radio_mode">Mode:</label>
                    <select id="radio_mode" name="airwave_settings[radio_mode]">
                        <option value="single" <?php selected($settings['radio_mode'] ?? 'single', 'single'); ?>>Single Stream</option>
                        <option value="rotate" <?php selected($settings['radio_mode'] ?? 'single', 'rotate'); ?>>Rotate Streams</option>
                    </select>
                </div>
                        
                <div class="airwave-inline">
                    <label for="rotate_interval">Rotation Interval (seconds):</label>
                    <input type="number" id="rotate_interval" name="airwave_settings[rotate_interval]" 
                           value="<?php echo esc_attr($settings['rotate_interval'] ?? 60); ?>" min="10" max="3600">
                        </div>
                        
                <div class="airwave-inline">
                    <label>
                        <input type="checkbox" name="airwave_settings[autoplay]" value="1" 
                               <?php checked(!empty($settings['autoplay'])); ?>>
                        Enable Autoplay
                    </label>
                </div>
            </div>
            
            <!-- Stream 1 (Always visible) -->
            <div class="airwave-stream-section" id="stream-section-1">
                <h4>Stream 1</h4>
                
                <div class="airwave-stream-fields">
                    <div class="airwave-field">
                        <label for="stream_1_name">Stream Name:</label>
                        <input type="text" id="stream_1_name" name="airwave_settings[stream_1_name]" 
                               value="<?php echo esc_attr($settings["stream_1_name"] ?? ''); ?>" 
                               placeholder="Stream 1">
                    </div>
                    
                    <div class="airwave-field">
                        <label for="stream_1_url">Stream URL:</label>
                        <input type="url" id="stream_1_url" name="airwave_settings[stream_1_url]" 
                               value="<?php echo esc_attr($settings["stream_1_url"] ?? ''); ?>" 
                               placeholder="https://stream.example.com:8000/stream">
                        </div>
                        
                    <div class="airwave-field">
                        <label for="stream_1_api_url">API URL:</label>
                        <input type="url" id="stream_1_api_url" name="airwave_settings[stream_1_api_url]" 
                               value="<?php echo esc_attr($settings["stream_1_api_url"] ?? ''); ?>" 
                               placeholder="https://stream.example.com:8000/cp/get_info.php?p=8000">
                            </div>
                            
                    <div class="airwave-field">
                        <button type="button" class="button airwave-test-stream" data-stream="1">Test Stream</button>
                        <span class="airwave-test-result" id="test-result-1"></span>
                    </div>
                            </div>
                        </div>
                        
            <!-- Additional Streams (Hidden by default) -->
            <?php for ($i = 2; $i <= 5; $i++): ?>
            <div class="airwave-stream-section" id="stream-section-<?php echo $i; ?>" style="display: none;">
                <h4>Stream <?php echo $i; ?> <button type="button" class="button button-small airwave-remove-stream" data-stream="<?php echo $i; ?>">Remove</button></h4>
                
                <div class="airwave-stream-fields">
                    <div class="airwave-field">
                        <label for="stream_<?php echo $i; ?>_name">Stream Name:</label>
                        <input type="text" id="stream_<?php echo $i; ?>_name" name="airwave_settings[stream_<?php echo $i; ?>_name]" 
                               value="<?php echo esc_attr($settings["stream_{$i}_name"] ?? ''); ?>" 
                               placeholder="Stream <?php echo $i; ?>">
                    </div>
                    
                    <div class="airwave-field">
                        <label for="stream_<?php echo $i; ?>_url">Stream URL:</label>
                        <input type="url" id="stream_<?php echo $i; ?>_url" name="airwave_settings[stream_<?php echo $i; ?>_url]" 
                               value="<?php echo esc_attr($settings["stream_{$i}_url"] ?? ''); ?>" 
                               placeholder="https://stream.example.com:8000/stream">
                        </div>
                        
                    <div class="airwave-field">
                        <label for="stream_<?php echo $i; ?>_api_url">API URL:</label>
                        <input type="url" id="stream_<?php echo $i; ?>_api_url" name="airwave_settings[stream_<?php echo $i; ?>_api_url]" 
                               value="<?php echo esc_attr($settings["stream_{$i}_api_url"] ?? ''); ?>" 
                               placeholder="https://stream.example.com:8000/cp/get_info.php?p=8000">
                        </div>
                        
                    <div class="airwave-field">
                        <button type="button" class="button airwave-test-stream" data-stream="<?php echo $i; ?>">Test Stream</button>
                        <span class="airwave-test-result" id="test-result-<?php echo $i; ?>"></span>
                    </div>
                </div>
            </div>
            <?php endfor; ?>
            
            <!-- Add Stream Button -->
            <div class="airwave-add-stream-section">
                <button type="button" class="button button-secondary" id="add-stream-btn">Add Stream</button>
                <span class="airwave-stream-count">Streams: <span id="stream-count">1</span>/5</span>
            </div>
            
            <div class="airwave-test-all">
                <button type="button" class="button button-primary airwave-test-all-streams">Test All Streams</button>
                <span class="airwave-test-result" id="test-all-result"></span>
            </div>

>
        </div>
        <?php
    }
    
    // Sonic Panel Configuration Tab
    private function render_sonic_tab($settings) {
        ?>
        <div class="airwave-tab-content">
            <h3>Sonic Panel Integration</h3>
            
            <!-- Sonic Panel Info -->
            <div class="airwave-info-notice" style="background: #f0f8ff; border: 1px solid #0073aa; border-radius: 4px; padding: 15px; margin-bottom: 20px;">
                <h4 style="margin-top: 0; color: #0073aa;">🎵 Sonic Panel Integration</h4>
                <p>Configure your Sonic Panel streaming service integration. This allows you to connect to Sonic Panel servers for enhanced radio streaming capabilities.</p>
            </div>
            
            <div class="airwave-sonic-config">
                <div class="airwave-field">
                    <label for="sonic_enabled">Enable Sonic Panel Integration:</label>
                    <input type="checkbox" id="sonic_enabled" name="airwave_settings[sonic_enabled]" value="1" <?php checked(isset($settings['sonic_enabled']) && $settings['sonic_enabled']); ?>>
                    <p class="description">Enable Sonic Panel integration for enhanced streaming features.</p>
                </div>
                
                <div class="airwave-field">
                    <label for="sonic_server_url">Sonic Server URL:</label>
                    <input type="url" id="sonic_server_url" name="airwave_settings[sonic_server_url]" value="<?php echo esc_attr($settings['sonic_server_url'] ?? 'https://sonic24.instainternet.com'); ?>" class="regular-text">
                    <p class="description">Base URL of your Sonic Panel server.</p>
                </div>
                
                <div class="airwave-field">
                    <label for="sonic_port">Sonic Port:</label>
                    <input type="number" id="sonic_port" name="airwave_settings[sonic_port]" value="<?php echo esc_attr($settings['sonic_port'] ?? '8416'); ?>" class="small-text">
                    <p class="description">Port number for your Sonic Panel server.</p>
                </div>
                
                <div class="airwave-field">
                    <label for="sonic_stream_url">Stream URL:</label>
                    <input type="url" id="sonic_stream_url" name="airwave_settings[sonic_stream_url]" value="<?php echo esc_attr($settings['sonic_stream_url'] ?? 'https://sonic24.instainternet.com/8416/stream'); ?>" class="regular-text">
                    <p class="description">Direct stream URL for audio playback.</p>
                </div>
                
                <div class="airwave-field">
                    <label for="sonic_json_api">JSON API URL:</label>
                    <input type="url" id="sonic_json_api" name="airwave_settings[sonic_json_api]" value="<?php echo esc_attr($settings['sonic_json_api'] ?? 'https://sonic24.instainternet.com/cp/get_info.php?p=8416'); ?>" class="regular-text">
                    <p class="description">JSON API endpoint for stream information.</p>
                </div>
                
                <div class="airwave-field">
                    <label for="sonic_admin_url">Admin Panel URL:</label>
                    <input type="url" id="sonic_admin_url" name="airwave_settings[sonic_admin_url]" value="<?php echo esc_attr($settings['sonic_admin_url'] ?? 'http://sonic24.instainternet.com:8416/admin.cgi'); ?>" class="regular-text">
                    <p class="description">Admin panel URL for Sonic Panel management.</p>
                </div>
                
                <div class="airwave-field">
                    <label for="sonic_username">Admin Username:</label>
                    <input type="text" id="sonic_username" name="airwave_settings[sonic_username]" value="<?php echo esc_attr($settings['sonic_username'] ?? 'admin'); ?>" class="regular-text">
                    <p class="description">Username for Sonic Panel admin access.</p>
                </div>
                
                <div class="airwave-field">
                    <label for="sonic_password">Admin Password:</label>
                    <input type="password" id="sonic_password" name="airwave_settings[sonic_password]" value="<?php echo esc_attr($settings['sonic_password'] ?? ''); ?>" class="regular-text">
                    <p class="description">Password for Sonic Panel admin access.</p>
                </div>
                
                <div class="airwave-field">
                    <label for="sonic_widget_script">Widget Script:</label>
                    <input type="url" id="sonic_widget_script" name="airwave_settings[sonic_widget_script]" value="<?php echo esc_attr($settings['sonic_widget_script'] ?? 'https://sonic24.instainternet.com/cp/widgets.js?r=684'); ?>" class="regular-text">
                    <p class="description">Widget script URL for Sonic Panel widgets.</p>
                </div>
                
                <!-- Test Sonic Panel Connection -->
                <div class="airwave-test-section">
                    <h4>Test Sonic Panel Connection</h4>
                    <button type="button" class="button button-primary airwave-test-sonic">Test Sonic Panel Connection</button>
                    <span class="airwave-test-result" id="test-sonic-result"></span>
                </div>
                
                <!-- Sonic Panel Status -->
                <div class="airwave-status-section">
                    <h4>Sonic Panel Status</h4>
                    <div class="airwave-status-indicator">
                        <span class="airwave-status-dot" id="sonic-status-dot"></span>
                        <span class="airwave-status-text" id="sonic-status-text">Not tested</span>
                    </div>
                </div>
            </div>
        </div>
        <?php
    }
    
    private function render_matrix_tab($settings) {
        ?>
        <div class="airwave-tab-content">
            <h3>Matrix Chat Configuration</h3>
            
            <!-- Version Update Notice -->
            <div class="airwave-update-notice" style="background: #e7f3ff; border: 1px solid #0073aa; border-radius: 4px; padding: 15px; margin-bottom: 20px;">
                <h4 style="margin-top: 0; color: #0073aa;">🆕 Version <?php echo AIRWAVE_PLUGIN_VERSION; ?> - Latest Update</h4>
                <ul style="margin-bottom: 0;">
                    <li><strong>✅ Enhanced Matrix Authentication</strong> - Fixed connection test to use authenticated credentials</li>
                    <li><strong>✅ Comprehensive Debug System</strong> - Added advanced testing and logging capabilities</li>
                    <li><strong>✅ Improved Connection Monitoring</strong> - Better status checks and error handling</li>
                    <li><strong>✅ Enhanced User Experience</strong> - Smoother UI updates and visual feedback</li>
                    <li><strong>✅ Advanced Testing Tools</strong> - Standard and comprehensive test modes with debug levels</li>
                    <li><strong>✅ Sonic Panel Integration</strong> - Added support for Sonic Panel streaming services</li>
                    <li><strong>✅ Disruptarian Radio Integration</strong> - Pre-configured Matrix room and credentials</li>
                </ul>
            </div>
            
            <div class="airwave-matrix-config">
                <div class="airwave-field">
                    <label for="matrix_homeserver">Homeserver:</label>
                    <select id="matrix_homeserver" name="airwave_settings[matrix_homeserver]">
                        <option value="https://matrix-client.matrix.org" <?php selected($settings['matrix_homeserver'] ?? 'https://matrix-client.matrix.org', 'https://matrix-client.matrix.org'); ?>>matrix.org</option>
                        <option value="https://matrix.element.io" <?php selected($settings['matrix_homeserver'] ?? '', 'https://matrix.element.io'); ?>>Element (matrix.org)</option>
                        <option value="custom" <?php selected($settings['matrix_homeserver'] ?? '', 'custom'); ?>>Custom Homeserver</option>
                </select>
            </div>
            
                <div class="airwave-field" id="custom-homeserver-field" style="<?php echo ($settings['matrix_homeserver'] ?? '') == 'custom' ? '' : 'display: none;'; ?>">
                    <label for="matrix_custom_homeserver">Custom Homeserver URL:</label>
                    <input type="url" id="matrix_custom_homeserver" name="airwave_settings[matrix_custom_homeserver]" 
                           value="<?php echo esc_attr($settings['matrix_custom_homeserver'] ?? ''); ?>" 
                           placeholder="https://your-homeserver.com">
            </div>
            
                <div class="airwave-field">
                    <label for="matrix_room_select">Recommended Rooms:</label>
                    <select id="matrix_room_select" class="regular-text">
                        <option value="">Select a room…</option>
                        <option value="#airwavesupport:matrix.org">#airwavesupport:matrix.org — Plugin Support From AirwaveAutomator</option>
                        <option value="#disruptiverequests:matrix.org">#disruptiverequests:matrix.org — Radio Chatbot Demonstration</option>
                        <option value="#matrix:matrix.org">#matrix:matrix.org — Official Matrix room</option>
                        <option value="#general:matrix.org">#general:matrix.org — General discussion</option>
                        <option value="#help:matrix.org">#help:matrix.org — Help and support</option>
                        <option value="manual">Enter manually…</option>
                    </select>
                    <p class="description">Choose a recommended room or select "Enter manually…" and type below.</p>
                </div>

                <div class="airwave-field">
                    <label for="matrix_room">Default Room:</label>
                    <input type="text" id="matrix_room" name="airwave_settings[matrix_room]" 
                           value="<?php echo esc_attr($settings['matrix_room'] ?? ''); ?>" 
                           placeholder="#general:matrix.org">
            </div>
            
                <div class="airwave-field">
                    <button type="button" class="button airwave-test-matrix">Test Matrix Connection</button>
                    <span class="airwave-test-result" id="test-matrix-result"></span>
                </div>
            </div>
            
            <div class="airwave-matrix-admin-login">
                <h4>Admin Login</h4>
                <p>Login with your Matrix credentials to enable admin chat functionality and status monitoring.</p>
                
                <div class="airwave-login-form">
                    <div class="airwave-field">
                        <label for="matrix_admin_username">Admin Username:</label>
                        <input type="text" id="matrix_admin_username" name="matrix_admin_username" 
                               value="<?php echo esc_attr($settings['matrix_admin_username'] ?? ''); ?>" 
                               placeholder="@username:matrix.org">
            </div>
                    
                    <div class="airwave-field">
                        <label for="matrix_admin_password">Admin Password:</label>
                        <input type="password" id="matrix_admin_password" name="matrix_admin_password" 
                               placeholder="Enter your Matrix password">
                    </div>
                    
                    <div class="airwave-field">
                        <button type="button" class="button button-primary" id="matrix-admin-login-btn">Login to Matrix</button>
                        <button type="button" class="button button-secondary" id="matrix-admin-logout-btn" style="display: none;">Logout</button>
                        <span class="airwave-test-result" id="matrix-login-result"></span>
                    </div>
                    
                    <!-- Live Connection Status -->
                    <div class="airwave-field" id="matrix-live-status" style="display: none;">
                        <div class="airwave-live-status">
                            <strong>Live Status:</strong> 
                            <span id="matrix-live-status-indicator" class="airwave-status-indicator offline"></span>
                            <span id="matrix-live-status-text">Checking connection...</span>
                            <small id="matrix-live-status-time"></small>
                        </div>
                    </div>
                    
                    <div class="airwave-field" id="matrix-login-status" style="display: none;">
                        <div class="airwave-login-status">
                            <strong>Connection Status:</strong> <span id="matrix-login-status-text" class="offline">Checking...</span><br>
                            <strong>User:</strong> <span id="matrix-login-user">-</span><br>
                            <strong>Homeserver:</strong> <span id="matrix-login-homeserver">-</span><br>
                            <small><em>Status is checked every 30 seconds to verify actual connection</em></small>
                        </div>
                    </div>
                </div>
            </div>
        </div>
        <?php
    }
    
    private function render_display_tab($settings) {
        ?>
        <div class="airwave-tab-content">
            <h3>Display Settings</h3>
            
            <div class="airwave-display-config">
                <div class="airwave-field">
                    <label for="stream_page_url">Stream Page URL:</label>
                    <input type="url" id="stream_page_url" name="airwave_settings[stream_page_url]" 
                           value="<?php echo esc_attr($settings['stream_page_url'] ?? ''); ?>" 
                           placeholder="https://yoursite.com/radio">
                    <p class="description">URL to the page containing the radio stream shortcode (used in top bar links).</p>
                </div>
                
                <div class="airwave-field">
                    <label for="chat_page_url">Chat Page URL:</label>
                    <input type="url" id="chat_page_url" name="airwave_settings[chat_page_url]" 
                           value="<?php echo esc_attr($settings['chat_page_url'] ?? ''); ?>" 
                           placeholder="https://yoursite.com/chat">
                    <p class="description">URL to the page containing the Matrix chat shortcode (used in top bar links).</p>
                </div>
                
                <div class="airwave-field">
                    <label for="refresh_interval">Status Refresh Interval (seconds):</label>
                    <input type="number" id="refresh_interval" name="airwave_settings[refresh_interval]" 
                           value="<?php echo esc_attr($settings['refresh_interval'] ?? 30); ?>" min="10" max="300">
                    <p class="description">How often to check stream and Matrix connection status (10-300 seconds).</p>
                </div>
                
                <div class="airwave-field">
                    <label>
                        <input type="checkbox" name="airwave_settings[show_status_bar]" value="1" 
                               <?php checked(!empty($settings['show_status_bar'])); ?>>
                        Show Status Bar
                    </label>
                    <p class="description">Display stream and Matrix status indicators in the WordPress admin bar.</p>
                </div>

                <div class="airwave-field">
                    <label for="display_theme">Theme:</label>
                    <select id="display_theme" name="airwave_settings[display_theme]">
                        <option value="dark" <?php selected($settings['display_theme'] ?? 'dark', 'dark'); ?>>Dark</option>
                        <option value="light" <?php selected($settings['display_theme'] ?? 'dark', 'light'); ?>>Light</option>
                        <option value="neutral" <?php selected($settings['display_theme'] ?? 'dark', 'neutral'); ?>>Modern Neutral</option>
                        <option value="system" <?php selected($settings['display_theme'] ?? 'dark', 'system'); ?>>System</option>
                    </select>
                    <p class="description">Choose the UI theme for Airwave widgets and shortcodes.</p>
                </div>

                <div class="airwave-field">
                    <label for="default_cover_image">Default Cover Image:</label>
                    <?php
                    $default_cover_id = absint($settings['default_cover_image'] ?? 0);
                    $default_cover_url = $default_cover_id ? wp_get_attachment_image_url($default_cover_id, 'medium') : '';
                    ?>
                    <div class="airwave-cover-image-preview" style="margin-bottom: 10px;">
                        <?php if ($default_cover_url): ?>
                            <img src="<?php echo esc_url($default_cover_url); ?>" style="max-width: 200px; height: auto; display: block; margin-bottom: 10px;" id="default_cover_preview" />
                        <?php else: ?>
                            <img src="" style="max-width: 200px; height: auto; display: none; margin-bottom: 10px;" id="default_cover_preview" />
                        <?php endif; ?>
                    </div>
                    <input type="hidden" id="default_cover_image" name="airwave_settings[default_cover_image]" value="<?php echo esc_attr($default_cover_id); ?>" />
                    <button type="button" class="button" id="upload_default_cover_btn"><?php echo $default_cover_id ? 'Change Image' : 'Upload Image'; ?></button>
                    <?php if ($default_cover_id): ?>
                        <button type="button" class="button" id="remove_default_cover_btn">Remove Image</button>
                    <?php endif; ?>
                    <p class="description">Upload a default cover image to display when album art is not available. Recommended size: 200x200px or larger square image.</p>
                </div>

                    <div class="airwave-field">
                        <label>Live Preview:</label>
                        <div id="airwave-theme-preview" class="airwave-combined-shortcode">
                            <div class="airwave-radio-content awmx-card">
                                <h4>Radio Widget Preview</h4>
                                <div class="awmx-inline">
                                    <button type="button" class="awmx-btn">Play</button>
                                    <input type="range" class="awmx-input" />
                                </div>
                            </div>
                            <div class="awmx-card">
                                <h4>Chat Panel Preview</h4>
                                <div class="awmx-inline">
                                    <input type="text" class="awmx-input" placeholder="Type a message..." />
                                    <button type="button" class="awmx-btn">Send</button>
                                </div>
                            </div>
                        </div>
                        <p class="description">Preview updates instantly when changing theme.</p>
                    </div>
                </div>

                <!-- Custom Front Row Content -->
                <div class="airwave-field">
                    <label for="custom_front_row">Custom Front Row Content:</label>
                    <textarea id="custom_front_row" name="airwave_settings[custom_front_row]" rows="6" class="large-text" placeholder="Paste text, HTML, or shortcodes here (e.g., [airwave_combined])"><?php echo esc_textarea($settings['custom_front_row'] ?? ''); ?></textarea>
                    <p class="description">This renders as a single full-width row. You can include HTML or any shortcode such as <code>[airwave_combined]</code>. Use the shortcode <code>[airwave_front_row]</code> on a page to display this area.</p>
                </div>
            </div>
        </div>
        <?php
    }
    
    // Matrix Chat Page
    public function matrix_page() {
        $settings = get_option('airwave_settings', array());
        $homeserver = $settings['matrix_homeserver'] ?? 'https://matrix-client.matrix.org';
        
        // Enqueue Matrix SDK and client scripts
        wp_enqueue_script(
            'matrix-sdk',
            AIRWAVE_PLUGIN_URL . 'third_party/matrix/browser-matrix.min.js',
            array(),
            AIRWAVE_PLUGIN_VERSION,
            true
        );
        
        wp_enqueue_script(
            'airwave-matrix-client',
            AIRWAVE_PLUGIN_URL . 'assets/js/awmx-matrix.js',
            array('matrix-sdk'),
            AIRWAVE_PLUGIN_VERSION,
            true
        );
        
        ?>
        <div class="wrap">
            <div class="airwave-header">
                <img src="<?php echo AIRWAVE_PLUGIN_URL; ?>assets/logo.png" alt="Airwave Audio Player Matrix" class="airwave-logo">
                <h1>Matrix Chat Client</h1>
            </div>
            
            <div class="airwave-admin-container">
                <div class="airwave-card">
                    <h3>Live Matrix Chat</h3>
                    <p>Use this live chat client to communicate with your Matrix community directly from the WordPress admin.</p>
                    
                    <div id="airwave-matrix-admin-client" class="awmx-matrix" data-homeserver="<?php echo esc_attr($homeserver); ?>">
                        <div class="awmx-matrix__root">
                            <form class="awmx-matrix__login">
                                <div class="awmx-stack">
                                    <div class="awmx-inline">
                                        <input type="text" name="username" class="awmx-input awmx-username" placeholder="@user:homeserver.tld">
                                        <input type="password" name="password" class="awmx-input awmx-password" placeholder="Password">
                                        <button type="submit" class="awmx-btn awmx-login">Login</button>
                                    </div>
                                    <div class="awmx-inline">
                                        <input type="text" class="awmx-input awmx-room-input" placeholder="#room:homeserver.tld">
                                        <button type="button" class="awmx-btn awmx-room-join">Join Room</button>
                                    </div>
                                </div>
                            </form>
                            <div class="awmx-matrix__app" hidden>
                                <div class="awmx-matrix__sidebar">
                                    <div class="awmx-room__header"><h4>Rooms</h4></div>
                                    <div class="awmx-room__list"></div>
                                    <div class="awmx-stack">
                                        <div class="awmx-inline">
                                            <input type="text" class="awmx-input awmx-join-input" placeholder="#room:homeserver.tld">
                                            <button type="button" class="awmx-btn awmx-join-btn">Join</button>
                                        </div>
                                    </div>
                                    <button type="button" class="awmx-btn awmx-logout">Logout</button>
                                </div>
                                <div class="awmx-matrix__timeline">
                                    <div class="awmx-room__header"><span class="awmx-room__name">Select a room</span></div>
                                    <div class="awmx-events" style="max-height: 360px; overflow: auto;"></div>
                                    <form class="awmx-composer">
                                        <div class="awmx-inline">
                                            <input type="text" class="awmx-input awmx-message" placeholder="Type a message">
                                            <button type="submit" class="awmx-btn">Send</button>
                                        </div>
                                    </form>
                                </div>
                            </div>
                        </div>
                    </div>
                </div>
            </div>
        </div>
        
        <script>
        jQuery(document).ready(function($) {
            console.log('Matrix SDK available:', typeof matrixcs !== 'undefined');
            console.log('Mount function available:', typeof window.awmxMount !== 'undefined');
            console.log('AWMX object available:', typeof AWMX !== 'undefined');
            
            // Prevent any form submission until mounted
            $('#airwave-matrix-admin-client form').on('submit', function(e) {
                e.preventDefault();
                e.stopPropagation();
                return false;
            });
            
            // Wait a bit for scripts to load, but also try immediately
            function tryMount() {
                if (typeof window.awmxMount === 'function') {
                    console.log('Initializing Matrix client...');
                    const root = document.getElementById('airwave-matrix-admin-client');
                    if (root) {
                        window.awmxMount(root);
                        return true;
                    }
                }
                return false;
            }
            
            // Try immediately first
            if (!tryMount()) {
                // If not ready, wait and retry
                setTimeout(function() {
                    if (!tryMount()) {
                        console.error('Matrix client mount function not available after delay');
                        console.log('Available globals:', Object.keys(window).filter(k => k.includes('matrix') || k.includes('Matrix') || k.includes('awmx')));
                        $('#airwave-matrix-admin-client').html('<p style="color: red; padding: 20px;">Error: Matrix client failed to load. Please check console for details and refresh the page.</p>');
                    }
                }, 1000);
            }
        });
        </script>
        <?php
    }
    
    // Shortcodes Page
    public function shortcodes_page() {
        ?>
        <div class="wrap">
            <div class="airwave-header">
                <img src="<?php echo AIRWAVE_PLUGIN_URL; ?>assets/logo.png" alt="Airwave Audio Player Matrix" class="airwave-logo">
                <h1>Shortcodes</h1>
            </div>
            
            <div class="airwave-admin-container">
                <div class="airwave-card">
                    <h3>Available Shortcodes</h3>
                    <p>Use these shortcodes to display Airwave Audio Player Matrix features on your posts and pages.</p>
                    
                    <div class="airwave-shortcode-section">
                        <h4>Radio Player Only</h4>
                        <div class="airwave-shortcode-example">
                    <code>[airwave_radio]</code>
                            <button type="button" class="button button-small airwave-copy-shortcode" data-shortcode="[airwave_radio]">Copy</button>
                        </div>
                        <div class="airwave-shortcode-example">
                            <code>[airwave_radio stream="0" show_history="true"]</code>
                            <button type="button" class="button button-small airwave-copy-shortcode" data-shortcode="[airwave_radio stream=&quot;0&quot; show_history=&quot;true&quot;]">Copy</button>
                        </div>
                        <p><strong>Parameters:</strong></p>
                        <ul>
                            <li><code>stream</code> - Stream index (0-4)</li>
                            <li><code>show_history</code> - Show song history (true/false)</li>
                            <li><code>title</code> - Custom title for the player</li>
                        </ul>
                    </div>
                    
                    <div class="airwave-shortcode-section">
                        <h4>Matrix Chat Only</h4>
                        <div class="airwave-shortcode-example">
                    <code>[airwave_matrix]</code>
                            <button type="button" class="button button-small airwave-copy-shortcode" data-shortcode="[airwave_matrix]">Copy</button>
                        </div>
                        <div class="airwave-shortcode-example">
                            <code>[airwave_matrix room="#general:matrix.org"]</code>
                            <button type="button" class="button button-small airwave-copy-shortcode" data-shortcode="[airwave_matrix room=&quot;#general:matrix.org&quot;]">Copy</button>
                        </div>
                        <p><strong>Parameters:</strong></p>
                        <ul>
                            <li><code>room</code> - Matrix room ID to join</li>
                            <li><code>title</code> - Custom title for the chat</li>
                        </ul>
                    </div>
                    
                    <div class="airwave-shortcode-section">
                        <h4>Combined Radio + Matrix Chat</h4>
                        <div class="airwave-shortcode-example">
                    <code>[airwave_combined]</code>
                            <button type="button" class="button button-small airwave-copy-shortcode" data-shortcode="[airwave_combined]">Copy</button>
                        </div>
                        <div class="airwave-shortcode-example">
                            <code>[airwave_combined stream="0" room="#general:matrix.org"]</code>
                            <button type="button" class="button button-small airwave-copy-shortcode" data-shortcode="[airwave_combined stream=&quot;0&quot; room=&quot;#general:matrix.org&quot;]">Copy</button>
                        </div>
                        <p><strong>Parameters:</strong></p>
                        <ul>
                            <li><code>stream</code> - Stream index (0-4)</li>
                            <li><code>room</code> - Matrix room ID to join</li>
                            <li><code>title</code> - Custom title for the combined widget</li>
                        </ul>
                    </div>
                    
                    <div class="airwave-shortcode-section">
                        <h4>Song History Only</h4>
                        <div class="airwave-shortcode-example">
                    <code>[airwave_song_history]</code>
                            <button type="button" class="button button-small airwave-copy-shortcode" data-shortcode="[airwave_song_history]">Copy</button>
                        </div>
                        <div class="airwave-shortcode-example">
                            <code>[airwave_song_history stream="0" limit="10"]</code>
                            <button type="button" class="button button-small airwave-copy-shortcode" data-shortcode="[airwave_song_history stream=&quot;0&quot; limit=&quot;10&quot;]">Copy</button>
                        </div>
                        <p><strong>Parameters:</strong></p>
                        <ul>
                            <li><code>stream</code> - Stream index (0-4)</li>
                            <li><code>limit</code> - Number of songs to show (2-20)</li>
                            <li><code>title</code> - Custom title for the history</li>
                        </ul>
                    </div>
                    
                    <div class="airwave-shortcode-section">
                        <h4>Audio Visualizer</h4>
                        <p>Display an interactive audio visualizer with play/pause controls and real-time visualization.</p>
                        <div class="airwave-shortcode-example">
                            <code>[airwave_visualizer stream="0" width="400" height="200" type="bars" title="Live Audio Visualizer"]</code>
                            <button class="airwave-copy-btn" data-text="[airwave_visualizer stream=&quot;0&quot; width=&quot;400&quot; height=&quot;200&quot; type=&quot;bars&quot; title=&quot;Live Audio Visualizer&quot;]">Copy</button>
                        </div>
                        <h5>Parameters:</h5>
                        <ul>
                            <li><strong>stream</strong> - Stream index (0-4, default: 0)</li>
                            <li><strong>width</strong> - Canvas width in pixels (default: 400)</li>
                            <li><strong>height</strong> - Canvas height in pixels (default: 200)</li>
                            <li><strong>type</strong> - Visualization type: bars, waveform, circle, spectrum (default: bars)</li>
                            <li><strong>title</strong> - Optional title above the visualizer</li>
                        </ul>
                        <h5>Features:</h5>
                        <ul>
                            <li>Real-time audio visualization using Web Audio API</li>
                            <li>Multiple visualization types (bars, waveform, circle, spectrum)</li>
                            <li>Play/pause controls with visual feedback</li>
                            <li>Volume control slider</li>
                            <li>Current song information display</li>
                            <li>Responsive design</li>
                        </ul>
                    </div>
                    
                    <div class="airwave-shortcode-section">
                        <h4>Multi-User Chat Room</h4>
                        <div class="airwave-shortcode-example">
                            <code>[airwave_chat_room]</code>
                            <button class="airwave-copy-btn" data-shortcode="[airwave_chat_room]">Copy</button>
                        </div>
                        <p>Creates a multi-user Matrix chat room where visitors can login and chat together.</p>
                        
                        <h5>Parameters:</h5>
                        <ul>
                            <li><code>room</code> - Matrix room ID (default: #airwavesupport:matrix.org)</li>
                            <li><code>title</code> - Custom title for the chat room</li>
                            <li><code>show_login</code> - Show login form (true/false, default: true)</li>
                            <li><code>max_height</code> - Maximum height of messages area in pixels (default: 400)</li>
                        </ul>
                        
                        <h5>Examples:</h5>
                        <div class="airwave-shortcode-example">
                            <code>[airwave_chat_room room="#general:matrix.org" title="Community Chat"]</code>
                            <button class="airwave-copy-btn" data-shortcode="[airwave_chat_room room=&quot;#general:matrix.org&quot; title=&quot;Community Chat&quot;]">Copy</button>
                        </div>
                        <div class="airwave-shortcode-example">
                            <code>[airwave_chat_room max_height="600" title="Support Chat"]</code>
                            <button class="airwave-copy-btn" data-shortcode="[airwave_chat_room max_height=&quot;600&quot; title=&quot;Support Chat&quot;]">Copy</button>
                        </div>
                        
                        <h5>Features:</h5>
                        <ul>
                            <li>Multi-user chat room functionality</li>
                            <li>Real-time messaging with Matrix SDK</li>
                            <li>User login/logout interface</li>
                            <li>Message history display</li>
                            <li>Typing indicators</li>
                            <li>User count display</li>
                            <li>Responsive design</li>
                    </ul>
                    </div>
                    
                    <div class="airwave-shortcode-section">
                        <h4>Song Ticker (Current + Scrolling History)</h4>
                        <div class="airwave-shortcode-example">
                            <code>[airwave_song_ticker]</code>
                            <button type="button" class="button button-small airwave-copy-shortcode" data-shortcode="[airwave_song_ticker]">Copy</button>
                        </div>
                        <div class="airwave-shortcode-example">
                            <code>[airwave_song_ticker stream="0" limit="10" scroll_speed="3" title="Now Playing"]</code>
                            <button type="button" class="button button-small airwave-copy-shortcode" data-shortcode="[airwave_song_ticker stream=&quot;0&quot; limit=&quot;10&quot; scroll_speed=&quot;3&quot; title=&quot;Now Playing&quot;]">Copy</button>
                        </div>
                        <p><strong>Parameters:</strong></p>
                        <ul>
                            <li><code>stream</code> - Stream index (0-4)</li>
                            <li><code>limit</code> - Number of songs to show (2-50)</li>
                            <li><code>scroll_speed</code> - Scroll speed in seconds (1-10)</li>
                            <li><code>title</code> - Custom title for the ticker widget</li>
                        </ul>
                        <p><strong>Features:</strong></p>
                        <ul>
                            <li><strong>Stock Ticker Effect:</strong> Continuous horizontal scrolling like financial tickers</li>
                            <li><strong>Current Song Display:</strong> Shows now playing with album art on the left</li>
                            <li><strong>Seamless Loop:</strong> Song history scrolls continuously without interruption</li>
                            <li><strong>Hover to Pause:</strong> Animation pauses when hovering over the ticker</li>
                            <li><strong>Responsive Design:</strong> Adapts to different screen sizes</li>
                            <li><strong>Configurable Speed:</strong> Adjustable scroll speed (1-10 seconds)</li>
                    </ul>
                    </div>
                </div>
            </div>
        </div>
        <?php
    }
    
    // Testing Page
    public function testing_page() {
        ?>
        <div class="wrap">
            <div class="airwave-header">
                <img src="<?php echo AIRWAVE_PLUGIN_URL; ?>assets/logo.png" alt="Airwave Audio Player Matrix" class="airwave-logo">
                <h1>Testing & Debug</h1>
            </div>
            
            <div class="airwave-testing-page">
                <h3>Plugin Testing & Debug System</h3>
                
                <!-- Test Configuration -->
                <div class="airwave-test-config">
                    <h4>Test Configuration</h4>
                    <table class="form-table">
                        <tr>
                            <th scope="row">Test Type</th>
                            <td>
                                <select id="test-type">
                                    <option value="standard">Standard Test</option>
                                    <option value="comprehensive">Comprehensive Test</option>
                                </select>
                                <p class="description">Standard: Basic functionality tests. Comprehensive: Detailed analysis with full logging.</p>
                            </td>
                        </tr>
                        <tr>
                            <th scope="row">Debug Level</th>
                            <td>
                                <select id="debug-level">
                                    <option value="minimal">Minimal (Errors Only)</option>
                                    <option value="standard">Standard (Errors + Warnings)</option>
                                    <option value="verbose">Verbose (All Actions)</option>
                                </select>
                                <p class="description">Controls the amount of debug information collected during tests.</p>
                            </td>
                        </tr>
                    </table>
                </div>
                
                <!-- Test Sections -->
                <div class="airwave-test-section">
                    <h4>Stream Testing</h4>
                    <button type="button" class="button button-primary airwave-test-all-streams">Test All Streams</button>
                    <span class="airwave-test-result" id="test-all-result"></span>
                </div>
                
                <div class="airwave-test-section">
                    <h4>Matrix Testing</h4>
                    <button type="button" class="button button-primary airwave-test-matrix">Test Matrix Connection</button>
                    <span class="airwave-test-result" id="test-matrix-result"></span>
                </div>
                
                <div class="airwave-test-section">
                    <h4>System Testing</h4>
                    <button type="button" class="button button-primary airwave-test-system">Test System Health</button>
                    <span class="airwave-test-result" id="test-system-result"></span>
                </div>
                
                <!-- Debug Log Display -->
                <div class="airwave-debug-section">
                    <h4>Debug Log</h4>
                    <div class="airwave-debug-controls">
                        <button type="button" class="button airwave-clear-log">Clear Log</button>
                        <button type="button" class="button airwave-download-log">Download Log</button>
                        <button type="button" class="button airwave-refresh-log">Refresh Log</button>
                    </div>
                    <div class="airwave-debug-log" id="debug-log">
                        <p>Debug log will appear here when tests are run...</p>
                    </div>
                </div>
                
                <!-- Debug Statistics -->
                <div class="airwave-debug-stats">
                    <h4>Debug Statistics</h4>
                    <div class="airwave-stats-grid">
                        <div class="airwave-stat-item">
                            <strong>Total Tests Run:</strong> <span id="total-tests">0</span>
                        </div>
                        <div class="airwave-stat-item">
                            <strong>Successful Tests:</strong> <span id="successful-tests">0</span>
                        </div>
                        <div class="airwave-stat-item">
                            <strong>Failed Tests:</strong> <span id="failed-tests">0</span>
                        </div>
                        <div class="airwave-stat-item">
                            <strong>Last Test Time:</strong> <span id="last-test-time">Never</span>
                        </div>
                    </div>
                </div>
            </div>
        </div>
        
        <style>
        .airwave-test-config {
            background: #f9f9f9;
            border: 1px solid #ddd;
            border-radius: 4px;
            padding: 20px;
            margin-bottom: 20px;
        }
        
        .airwave-test-section {
            background: white;
            border: 1px solid #ddd;
            border-radius: 4px;
            padding: 20px;
            margin-bottom: 20px;
        }
        
        .airwave-test-section h4 {
            margin-top: 0;
            color: #0073aa;
        }
        
        .airwave-test-result {
            margin-left: 10px;
            font-weight: bold;
        }
        
        .airwave-test-result.success {
            color: #46b450;
        }
        
        .airwave-test-result.error {
            color: #dc3232;
        }
        
        .airwave-debug-section {
            background: #f0f0f1;
            border: 1px solid #ddd;
            border-radius: 4px;
            padding: 20px;
            margin-bottom: 20px;
        }
        
        .airwave-debug-controls {
            margin-bottom: 15px;
        }
        
        .airwave-debug-controls .button {
            margin-right: 10px;
        }
        
        .airwave-debug-log {
            background: #1e1e1e;
            color: #ffffff;
            font-family: 'Courier New', monospace;
            font-size: 12px;
            padding: 15px;
            border-radius: 4px;
            max-height: 400px;
            overflow-y: auto;
            white-space: pre-wrap;
            word-wrap: break-word;
        }
        
        .airwave-debug-stats {
            background: white;
            border: 1px solid #ddd;
            border-radius: 4px;
            padding: 20px;
        }
        
        .airwave-stats-grid {
            display: grid;
            grid-template-columns: repeat(auto-fit, minmax(200px, 1fr));
            gap: 15px;
        }
        
        .airwave-stat-item {
            background: #f9f9f9;
            padding: 10px;
            border-radius: 4px;
            border-left: 4px solid #0073aa;
        }
        
        .debug-entry {
            margin-bottom: 5px;
        }
        
        .debug-entry.error {
            color: #ff6b6b;
        }
        
        .debug-entry.warning {
            color: #ffd93d;
        }
        
        .debug-entry.success {
            color: #6bcf7f;
        }
        
        .debug-entry.info {
            color: #74c0fc;
        }
        </style>
        <?php
    }
    
    // Documentation Page
    public function documentation_page() {
        ?>
        <div class="wrap">
            <div class="airwave-header">
                <img src="<?php echo AIRWAVE_PLUGIN_URL; ?>assets/logo.png" alt="Airwave Audio Player Matrix" class="airwave-logo">
                <h1>Documentation</h1>
            </div>
            
            <div class="airwave-documentation-page">
                <h3>Plugin Documentation</h3>
                <p>Complete documentation and support information.</p>
                
                <div class="airwave-doc-section">
                    <h4>Installation</h4>
                    <p>1. Upload plugin files to WordPress</p>
                    <p>2. Activate the plugin</p>
                    <p>3. Configure settings</p>
                </div>
                
                <div class="airwave-doc-section">
                    <h4>Usage</h4>
                    <p>Use shortcodes to display radio player and Matrix chat on your pages.</p>
                </div>
                
                <div class="airwave-doc-section">
                    <h4>Recommended Rooms</h4>
                    <ul>
                        <li><code>#airwavesupport:matrix.org</code> | Plugin Support From AirwaveAutomator</li>
                        <li><code>#disruptiverequests:matrix.org</code> | Radio Chatbot Demonstration for Matrix chat rooms</li>
                    </ul>
                </div>

                <div class="airwave-doc-section">
                    <h4>Popular Matrix Rooms</h4>
                    <ul>
                        <li><code>#matrix:matrix.org</code> - Official Matrix room</li>
                        <li><code>#general:matrix.org</code> - General discussion</li>
                        <li><code>#help:matrix.org</code> - Help and support</li>
                    </ul>
                </div>

                <div class="airwave-doc-section">
                    <h4>Support</h4>
                    <ul>
                        <li><strong>Support user/bot on MATRIX Chat:</strong> <code>@disruptarianradio:matrix.org</code></li>
                        <li><strong>Server:</strong> <code>matrix.org</code></li>
                        <li><strong>Channel:</strong> <code>#airwavesupport:matrix.org</code></li>
                        <li><strong>Toll Free:</strong> +1 (888) 264-6790</li>
                        <li><strong>Website:</strong> <a href="https://spunwebtechnology.com" target="_blank">spunwebtechnology.com</a></li>
                        <li><strong>Email:</strong> <a href="mailto:support@spunwebtechnology.com">support@spunwebtechnology.com</a></li>
                    </ul>
                </div>
            </div>
        </div>
        <?php
    }
    
    // Clear Cache Page
    public function clear_cache_page() {
        // Handle cache clearing if requested
        if (isset($_POST['clear_cache']) && wp_verify_nonce($_POST['_wpnonce'], 'clear_cache')) {
            // Clear WordPress object cache
            if (function_exists('wp_cache_flush')) {
                wp_cache_flush();
            }
            
            // Clear WordPress transients
            global $wpdb;
            $wpdb->query("DELETE FROM {$wpdb->options} WHERE option_name LIKE '_transient_%'");
            $wpdb->query("DELETE FROM {$wpdb->options} WHERE option_name LIKE '_site_transient_%'");
            
            // Clear WordPress plugin cache
            if (function_exists('wp_clean_plugins_cache')) {
                wp_clean_plugins_cache();
            }
            
            // Clear custom plugin caches
            delete_option('airwave_plugin_version_cache');
            delete_transient('airwave_plugin_info');
            delete_option('airwave_last_version_check');
            
            echo '<div class="notice notice-success"><p>✅ All caches cleared successfully!</p></div>';
        }
        
        ?>
        <div class="wrap">
            <div class="airwave-header">
                <img src="<?php echo AIRWAVE_PLUGIN_URL; ?>assets/logo.png" alt="Airwave Audio Player Matrix" class="airwave-logo">
                <h1>Clear Cache</h1>
            </div>
            
            <div class="airwave-clear-cache-page">
                <h3>WordPress Cache Management</h3>
                <p>Use this page to clear WordPress caches and force the plugin to refresh its version information.</p>
                
                        <div class="airwave-cache-info">
                            <h4>Current Plugin Information</h4>
                            <p><strong>Plugin Version:</strong> <?php echo AIRWAVE_PLUGIN_VERSION; ?></p>
                            <p><strong>Cache Buster:</strong> 2025-01-19-00:00</p>
                            <p><strong>Last Version Check:</strong> <?php echo get_option('airwave_last_version_check', 'Not set'); ?></p>
                        </div>
                
                <form method="post" action="">
                    <?php wp_nonce_field('clear_cache'); ?>
                    <p>
                        <input type="submit" name="clear_cache" class="button button-primary" value="Clear All Caches" 
                               onclick="return confirm('Are you sure you want to clear all WordPress caches?');">
                    </p>
                </form>
                
                <div class="airwave-cache-help">
                    <h4>What This Does:</h4>
                    <ul>
                        <li>Clears WordPress object cache</li>
                        <li>Removes all transients and site transients</li>
                        <li>Cleans plugin cache</li>
                        <li>Forces WordPress to re-read plugin headers</li>
                        <li>Updates version information</li>
                    </ul>
                    
                            <p><strong>Note:</strong> After clearing caches, refresh your browser to see the updated version (<?php echo AIRWAVE_PLUGIN_VERSION; ?>).</p>
                </div>
            </div>
        </div>
        <?php
    }
    
    // AJAX Handlers
    public function save_settings() {
        check_ajax_referer('airwave_nonce', 'nonce');
        
        if (!current_user_can('manage_options')) {
            wp_send_json_error('Insufficient permissions');
        }
        
        // Get the submitted settings
        $settings = $_POST['airwave_settings'] ?? array();
        
        // Sanitize the settings
        $sanitized_settings = $this->sanitize_settings($settings);
        
        // Save the settings
        update_option('airwave_settings', $sanitized_settings);
        
        wp_send_json_success('Settings saved successfully');
    }
    
    public function test_stream() {
        check_ajax_referer('airwave_nonce', 'nonce');
        
        $stream_index = intval($_POST['stream']);
        $settings = get_option('airwave_settings', array());
        
        $stream_url = $settings["stream_{$stream_index}_url"] ?? '';
        if (empty($stream_url)) {
            wp_send_json_error('No stream URL configured');
        }
        
        // Test stream connectivity
        $context = stream_context_create(array(
            'http' => array(
                'timeout' => 10,
                'user_agent' => 'Airwave/1.0'
            )
        ));
        
        $headers = @get_headers($stream_url, 1, $context);
        if ($headers && strpos($headers[0], '200') !== false) {
            wp_send_json_success("Stream {$stream_index}: Online");
        } else {
            wp_send_json_error("Stream {$stream_index}: Offline");
        }
    }
    
    public function test_all_streams() {
        check_ajax_referer('airwave_nonce', 'nonce');
        
        $settings = get_option('airwave_settings', array());
        $results = array();
        
        for ($i = 1; $i <= 5; $i++) {
            $stream_url = $settings["stream_{$i}_url"] ?? '';
            if (!empty($stream_url)) {
                $context = stream_context_create(array(
                    'http' => array(
                        'timeout' => 10,
                        'user_agent' => 'Airwave/1.0'
                    )
                ));
                
                $headers = @get_headers($stream_url, 1, $context);
                if ($headers && strpos($headers[0], '200') !== false) {
                    $results[] = "Stream {$i}: Online";
                } else {
                    $results[] = "Stream {$i}: Offline";
                }
            }
        }
        
        if (empty($results)) {
            wp_send_json_error('No streams configured');
        } else {
            wp_send_json_success(implode('<br>', $results));
        }
    }
    
    public function test_sonic() {
        check_ajax_referer('airwave_nonce', 'nonce');
        
        $settings = get_option('airwave_settings', array());
        $sonic_enabled = !empty($settings['sonic_enabled']);
        
        // Allow testing if credentials/URLs are present even if the toggle is off
        $sonic_stream_url = trim($settings['sonic_stream_url'] ?? '');
        $sonic_json_api = trim($settings['sonic_json_api'] ?? '');
        $sonic_admin_url = trim($settings['sonic_admin_url'] ?? '');
        
        if (!$sonic_enabled && empty($sonic_stream_url) && empty($sonic_json_api)) {
            wp_send_json_error('Sonic Panel integration is not enabled. Enable it in settings or provide Stream/API URL.');
        }
        
        if (empty($sonic_stream_url)) {
            wp_send_json_error('Sonic Panel stream URL is not configured');
        }
        
        $results = array();
        
        // Test stream URL
        $context = stream_context_create(array(
            'http' => array(
                'timeout' => 10,
                'user_agent' => 'Airwave/' . AIRWAVE_PLUGIN_VERSION,
                'method' => 'GET'
            )
        ));
        
        $stream_response = @file_get_contents($sonic_stream_url, false, $context);
        if ($stream_response !== false) {
            $results['stream'] = 'Stream URL accessible';
        } else {
            $results['stream'] = 'Stream URL not accessible';
        }
        
        // Test JSON API
        if (!empty($sonic_json_api)) {
            $json_response = @file_get_contents($sonic_json_api, false, $context);
            if ($json_response !== false) {
                $json_data = json_decode($json_response, true);
                if ($json_data !== null) {
                    $results['json_api'] = 'JSON API accessible and valid';
                    if (isset($json_data['server_name'])) {
                        $results['server_name'] = $json_data['server_name'];
                    }
                    if (isset($json_data['current_song'])) {
                        $results['current_song'] = $json_data['current_song'];
                    }
                    if (isset($json_data['listeners'])) {
                        $results['listeners'] = $json_data['listeners'];
                    }
                } else {
                    $results['json_api'] = 'JSON API accessible but invalid response';
                }
            } else {
                $results['json_api'] = 'JSON API not accessible';
            }
        }
        
        // Test admin panel
        if (!empty($sonic_admin_url)) {
            $admin_response = @file_get_contents($sonic_admin_url, false, $context);
            if ($admin_response !== false) {
                $results['admin_panel'] = 'Admin panel accessible';
            } else {
                $results['admin_panel'] = 'Admin panel not accessible';
            }
        }
        
        $success = isset($results['stream']) && strpos($results['stream'], 'accessible') !== false;
        
        if ($success) {
            wp_send_json_success('Sonic Panel Connection: Online - ' . implode(', ', $results));
        } else {
            wp_send_json_error('Sonic Panel Connection: Offline - ' . implode(', ', $results));
        }
    }
    
    public function test_matrix() {
        check_ajax_referer('airwave_nonce', 'nonce');
        
        $settings = get_option('airwave_settings', array());
        $homeserver = $settings['matrix_homeserver'] ?? 'https://matrix-client.matrix.org';
        
        if ($homeserver === 'custom') {
            $homeserver = $settings['matrix_custom_homeserver'] ?? '';
        }
        
        if (empty($homeserver)) {
            wp_send_json_error('No homeserver configured');
        }
        
        // Check if admin is logged in with valid credentials
        $admin_login_data = get_user_meta(get_current_user_id(), 'airwave_matrix_admin_login', true);
        
        if (!empty($admin_login_data) && !empty($admin_login_data['access_token'])) {
            // Test actual Matrix connection with access token
            $is_connected = $this->verify_matrix_connection($admin_login_data);
            
            if ($is_connected) {
                wp_send_json_success('Matrix Connection: Online (Authenticated as ' . ($admin_login_data['user_id'] ?? 'Unknown') . ')');
            } else {
                // Try to refresh the connection with stored credentials
                $refresh_result = $this->refresh_matrix_connection($admin_login_data);
                if ($refresh_result['success']) {
                    wp_send_json_success('Matrix Connection: Online (Re-authenticated as ' . ($refresh_result['user_id'] ?? 'Unknown') . ')');
                } else {
                    wp_send_json_error('Matrix Connection: Offline (Authentication failed - ' . ($refresh_result['error'] ?? 'Token expired') . ')');
                }
            }
        } else {
            // Test basic homeserver connectivity
            $context = stream_context_create(array(
                'http' => array(
                    'timeout' => 10,
                    'user_agent' => 'Airwave/1.0'
                )
            ));
            
            $test_url = rtrim($homeserver, '/') . '/_matrix/client/versions';
            $headers = @get_headers($test_url, 1, $context);
            
            if ($headers && strpos($headers[0], '200') !== false) {
                wp_send_json_success('Matrix Connection: Online (Homeserver accessible, but admin not logged in)');
            } else {
                wp_send_json_error('Matrix Connection: Offline (Cannot reach homeserver)');
            }
        }
    }
    
    public function get_status() {
        check_ajax_referer('airwave_status_nonce', 'nonce');
        
        $settings = get_option('airwave_settings', array());
        
        // Check stream status
        $stream_online = false;
        for ($i = 1; $i <= 5; $i++) {
            $stream_url = $settings["stream_{$i}_url"] ?? '';
            if (!empty($stream_url)) {
                // Prefer WordPress HTTP API with redirect handling; fallback to GET if HEAD fails
                $response = wp_remote_head($stream_url, array(
                    'timeout' => 8,
                    'redirection' => 10,
                    'headers' => array(
                        'User-Agent' => 'Airwave/1.0',
                        'Icy-MetaData' => '1',
                        'Accept' => '*/*',
                    ),
                    'sslverify' => false,
                ));
                if (is_wp_error($response)) {
                    $response = wp_remote_get($stream_url, array(
                        'timeout' => 8,
                        'redirection' => 10,
                        'headers' => array(
                            'User-Agent' => 'Airwave/1.0',
                            'Icy-MetaData' => '1',
                            'Accept' => '*/*',
                        ),
                        'sslverify' => false,
                    ));
                }
                if (!is_wp_error($response)) {
                    $code = wp_remote_retrieve_response_code($response);
                    $ctype = wp_remote_retrieve_header($response, 'content-type');
                    if (($code >= 200 && $code < 400) || (is_string($ctype) && (stripos($ctype, 'audio') !== false || stripos($ctype, 'octet-stream') !== false))) {
                        $stream_online = true;
                        break;
                    }
                }
            }
        }
        
        // Check Matrix login status with actual connection verification
        $matrix_logged_in = false;
        $matrix_user = 'Matrix';
        $admin_login_data = get_user_meta(get_current_user_id(), 'airwave_matrix_admin_login', true);
        
        if (!empty($admin_login_data) && !empty($admin_login_data['logged_in'])) {
            // Verify actual Matrix connection
            $matrix_logged_in = $this->verify_matrix_connection($admin_login_data);
            $matrix_user = $admin_login_data['username'] ?? 'Matrix';
            
            // If connection is lost, update the stored status
            if (!$matrix_logged_in) {
                $admin_login_data['logged_in'] = false;
                $admin_login_data['last_check'] = time();
                update_user_meta(get_current_user_id(), 'airwave_matrix_admin_login', $admin_login_data);
            }
        }
        
        wp_send_json_success(array(
            'stream_online' => $stream_online,
            'stream_url' => $settings['stream_page_url'] ?? home_url('/radio/'),
            'matrix_logged_in' => $matrix_logged_in,
            'matrix_user' => $matrix_user,
            'chat_url' => $settings['chat_page_url'] ?? home_url('/chat/')
        ));
    }
    
    public function matrix_login() {
        check_ajax_referer('airwave_matrix_login', 'nonce');
        
        $homeserver = sanitize_url($_POST['homeserver'] ?? '');
        $username = sanitize_text_field($_POST['username'] ?? '');
        $password = isset($_POST['password']) ? $_POST['password'] : ''; // Don't sanitize password
        $display_name = sanitize_text_field($_POST['display_name'] ?? '');
        
        if (empty($homeserver) || empty($username) || empty($password)) {
            wp_send_json_error('Missing required fields');
        }
        
        // Store login credentials in session/transient for this user
        $login_data = array(
            'homeserver' => $homeserver,
            'username' => $username,
            'password' => $password,
            'display_name' => $display_name,
            'logged_in' => true,
            'login_time' => time()
        );
        
        // Store in user meta for persistence
        update_user_meta(get_current_user_id(), 'airwave_matrix_login', $login_data);
        
        wp_send_json_success('Login successful');
    }
    
    public function matrix_logout() {
        check_ajax_referer('airwave_matrix_logout', 'nonce');
        
        // Clear stored login data
        delete_user_meta(get_current_user_id(), 'airwave_matrix_login');
        
        wp_send_json_success('Logout successful');
    }
    
    public function matrix_admin_login() {
        check_ajax_referer('airwave_nonce', 'nonce');
        
        if (!current_user_can('manage_options')) {
            wp_send_json_error('Insufficient permissions');
        }
        
        $username = sanitize_text_field($_POST['username']);
        $password = $_POST['password']; // Don't sanitize password
        
        if (empty($username) || empty($password)) {
            wp_send_json_error('Username and password are required');
        }
        
        // Get Matrix settings
        $settings = get_option('airwave_settings', array());
        $homeserver = $settings['matrix_homeserver'] ?? 'https://matrix-client.matrix.org';
        
        if ($homeserver === 'custom') {
            $homeserver = $settings['matrix_custom_homeserver'] ?? 'https://matrix-client.matrix.org';
        }
        
        // Perform actual Matrix login to get access token
        $login_result = $this->perform_matrix_login($homeserver, $username, $password);
        
        if ($login_result['success']) {
            // Store admin login data with access token
            $admin_login_data = array(
                'logged_in' => true,
                'username' => $username,
                'homeserver' => $homeserver,
                'access_token' => $login_result['access_token'],
                'user_id' => $login_result['user_id'],
                'device_id' => $login_result['device_id'],
                'login_time' => time(),
                'last_check' => time()
            );
            
            update_user_meta(get_current_user_id(), 'airwave_matrix_admin_login', $admin_login_data);
            
            wp_send_json_success(array(
                'user_id' => $login_result['user_id'],
                'homeserver' => $homeserver
            ));
        } else {
            wp_send_json_error($login_result['error']);
        }
    }
    
    public function matrix_admin_logout() {
        check_ajax_referer('airwave_nonce', 'nonce');
        
        if (!current_user_can('manage_options')) {
            wp_send_json_error('Insufficient permissions');
        }
        
        // Clear admin login data
        delete_user_meta(get_current_user_id(), 'airwave_matrix_admin_login');
        
        wp_send_json_success('Admin logout successful');
    }
    
    public function matrix_admin_status() {
        check_ajax_referer('airwave_nonce', 'nonce');
        
        if (!current_user_can('manage_options')) {
            wp_send_json_error('Insufficient permissions');
        }
        
        $admin_login_data = get_user_meta(get_current_user_id(), 'airwave_matrix_admin_login', true);
        
        if (!empty($admin_login_data) && !empty($admin_login_data['logged_in'])) {
            // Verify actual connection
            $is_connected = $this->verify_matrix_connection($admin_login_data);
            
            wp_send_json_success(array(
                'logged_in' => $is_connected,
                'user_id' => $admin_login_data['username'],
                'homeserver' => $admin_login_data['homeserver']
            ));
        } else {
            wp_send_json_success(array(
                'logged_in' => false
            ));
        }
    }
    
    /**
     * Perform actual Matrix login to get access token
     */
    private function perform_matrix_login($homeserver, $username, $password) {
        $login_url = rtrim($homeserver, '/') . '/_matrix/client/r0/login';
        
        $login_data = array(
            'type' => 'm.login.password',
            'identifier' => array(
                'type' => 'm.id.user',
                'user' => $username
            ),
            'password' => $password,
            'initial_device_display_name' => 'Airwave WordPress Admin'
        );
        
        $context = stream_context_create(array(
            'http' => array(
                'method' => 'POST',
                'header' => 'Content-Type: application/json',
                'content' => json_encode($login_data),
                'timeout' => 30,
                'user_agent' => 'Airwave/1.0'
            )
        ));
        
        $response = @file_get_contents($login_url, false, $context);
        
        if ($response === false) {
            return array(
                'success' => false,
                'error' => 'Failed to connect to Matrix homeserver'
            );
        }
        
        $result = json_decode($response, true);
        
        if (isset($result['access_token'])) {
            return array(
                'success' => true,
                'access_token' => $result['access_token'],
                'user_id' => $result['user_id'],
                'device_id' => $result['device_id']
            );
        } else {
            $error = $result['error'] ?? 'Unknown login error';
            return array(
                'success' => false,
                'error' => 'Matrix login failed: ' . $error
            );
        }
    }
    
    /**
     * Verify actual Matrix connection by testing API endpoints
     */
    private function verify_matrix_connection($login_data) {
        if (empty($login_data) || empty($login_data['homeserver']) || empty($login_data['access_token'])) {
            return false;
        }
        
        $homeserver = $login_data['homeserver'];
        $access_token = $login_data['access_token'];
        
        // Test Matrix whoami endpoint with access token
        $whoami_url = rtrim($homeserver, '/') . '/_matrix/client/r0/account/whoami';
        
        $context = stream_context_create(array(
            'http' => array(
                'method' => 'GET',
                'header' => 'Authorization: Bearer ' . $access_token,
                'timeout' => 10,
                'user_agent' => 'Airwave/1.0'
            )
        ));
        
        $response = @file_get_contents($whoami_url, false, $context);
        
        if ($response === false) {
            return false;
        }
        
        $result = json_decode($response, true);
        
        // If we get a valid response with user_id, the connection is good
        return isset($result['user_id']);
    }
    
    // Debug System Methods
    public function test_system() {
        check_ajax_referer('airwave_nonce', 'nonce');
        
        if (!current_user_can('manage_options')) {
            wp_send_json_error('Insufficient permissions');
        }
        
        $test_type = sanitize_text_field($_POST['test_type'] ?? 'standard');
        $debug_level = sanitize_text_field($_POST['debug_level'] ?? 'standard');
        
        $this->log_debug('info', "Starting {$test_type} system test with {$debug_level} debug level");
        
        $results = array();
        
        // Test WordPress environment
        $results['wordpress'] = $this->test_wordpress_environment($debug_level);
        
        // Test plugin configuration
        $results['plugin'] = $this->test_plugin_configuration($debug_level);
        
        // Test Matrix connection
        $results['matrix'] = $this->test_matrix_debug($debug_level);
        
        // Test stream configuration
        $results['streams'] = $this->test_stream_configuration($debug_level);
        
        // Calculate overall health
        $overall_health = $this->calculate_system_health($results);
        
        $this->log_debug('info', "System test completed. Overall health: {$overall_health}%");
        
        wp_send_json_success(array(
            'overall_health' => $overall_health,
            'results' => $results,
            'debug_log' => $this->get_debug_log_content()
        ));
    }
    
    private function test_wordpress_environment($debug_level) {
        $this->log_debug('info', 'Testing WordPress environment');
        
        $tests = array();
        
        // Test WordPress version
        global $wp_version;
        $tests['wp_version'] = array(
            'status' => version_compare($wp_version, '5.0', '>=') ? 'pass' : 'fail',
            'message' => "WordPress version: {$wp_version}",
            'value' => $wp_version
        );
        
        // Test PHP version
        $php_version = PHP_VERSION;
        $tests['php_version'] = array(
            'status' => version_compare($php_version, '7.4', '>=') ? 'pass' : 'fail',
            'message' => "PHP version: {$php_version}",
            'value' => $php_version
        );
        
        // Test memory limit
        $memory_limit = ini_get('memory_limit');
        $tests['memory_limit'] = array(
            'status' => $this->parse_memory_limit($memory_limit) >= 128 ? 'pass' : 'warning',
            'message' => "Memory limit: {$memory_limit}",
            'value' => $memory_limit
        );
        
        // Test file permissions
        $upload_dir = wp_upload_dir();
        $tests['upload_permissions'] = array(
            'status' => is_writable($upload_dir['basedir']) ? 'pass' : 'fail',
            'message' => "Upload directory writable: " . ($upload_dir['basedir']),
            'value' => $upload_dir['basedir']
        );
        
        $this->log_debug('info', 'WordPress environment test completed');
        
        return $tests;
    }
    
    private function test_plugin_configuration($debug_level) {
        $this->log_debug('info', 'Testing plugin configuration');
        
        $tests = array();
        $settings = get_option('airwave_settings', array());
        
        // Test plugin version
        $tests['plugin_version'] = array(
            'status' => 'pass',
            'message' => "Plugin version: " . AIRWAVE_PLUGIN_VERSION,
            'value' => AIRWAVE_PLUGIN_VERSION
        );
        
        // Test settings existence
        $tests['settings_exist'] = array(
            'status' => !empty($settings) ? 'pass' : 'warning',
            'message' => 'Settings configured: ' . (empty($settings) ? 'No' : 'Yes'),
            'value' => !empty($settings)
        );
        
        // Test Matrix configuration
        $matrix_homeserver = $settings['matrix_homeserver'] ?? '';
        $tests['matrix_homeserver'] = array(
            'status' => !empty($matrix_homeserver) ? 'pass' : 'warning',
            'message' => 'Matrix homeserver: ' . ($matrix_homeserver ?: 'Not configured'),
            'value' => $matrix_homeserver
        );
        
        $this->log_debug('info', 'Plugin configuration test completed');
        
        return $tests;
    }
    
    private function test_matrix_debug($debug_level) {
        $this->log_debug('info', 'Testing Matrix connection with debug');
        
        $tests = array();
        $settings = get_option('airwave_settings', array());
        $homeserver = $settings['matrix_homeserver'] ?? 'https://matrix-client.matrix.org';
        
        if ($homeserver === 'custom') {
            $homeserver = $settings['matrix_custom_homeserver'] ?? '';
        }
        
        // Test homeserver URL format
        $tests['homeserver_url'] = array(
            'status' => filter_var($homeserver, FILTER_VALIDATE_URL) ? 'pass' : 'fail',
            'message' => 'Homeserver URL format: ' . ($homeserver ?: 'Not configured'),
            'value' => $homeserver
        );
        
        // Test admin login status
        $admin_login_data = get_user_meta(get_current_user_id(), 'airwave_matrix_admin_login', true);
        $tests['admin_login'] = array(
            'status' => !empty($admin_login_data['access_token']) ? 'pass' : 'warning',
            'message' => 'Admin logged in: ' . (!empty($admin_login_data['access_token']) ? 'Yes' : 'No'),
            'value' => !empty($admin_login_data['access_token'])
        );
        
        if (!empty($admin_login_data['access_token'])) {
            // Test authenticated connection
            $is_connected = $this->verify_matrix_connection($admin_login_data);
            $tests['authenticated_connection'] = array(
                'status' => $is_connected ? 'pass' : 'fail',
                'message' => 'Authenticated connection: ' . ($is_connected ? 'Success' : 'Failed'),
                'value' => $is_connected
            );
            
            if ($is_connected) {
                $this->log_debug('success', 'Matrix authenticated connection successful');
            } else {
                $this->log_debug('error', 'Matrix authenticated connection failed');
            }
        } else {
            // Test basic homeserver connectivity
            $context = stream_context_create(array(
                'http' => array(
                    'timeout' => 10,
                    'user_agent' => 'Airwave/' . AIRWAVE_PLUGIN_VERSION,
                    'method' => 'GET'
                )
            ));
            
            $test_url = rtrim($homeserver, '/') . '/_matrix/client/versions';
            $this->log_debug('info', "Testing homeserver connectivity: {$test_url}");
            
            $response = @file_get_contents($test_url, false, $context);
            $tests['homeserver_connectivity'] = array(
                'status' => $response !== false ? 'pass' : 'fail',
                'message' => 'Homeserver connectivity: ' . ($response !== false ? 'Success' : 'Failed'),
                'value' => $response !== false
            );
            
            if ($response !== false) {
                $this->log_debug('success', 'Homeserver connectivity successful');
            } else {
                $this->log_debug('error', 'Homeserver connectivity failed');
            }
        }
        
        $this->log_debug('info', 'Matrix connection test completed');
        
        return $tests;
    }
    
    private function test_stream_configuration($debug_level) {
        $this->log_debug('info', 'Testing stream configuration');
        
        $tests = array();
        $settings = get_option('airwave_settings', array());
        
        $stream_count = 0;
        for ($i = 1; $i <= 5; $i++) {
            $stream_key = "stream_{$i}_url";
            if (!empty($settings[$stream_key])) {
                $stream_count++;
                $tests["stream_{$i}"] = array(
                    'status' => 'pass',
                    'message' => "Stream {$i} configured",
                    'value' => $settings[$stream_key]
                );
            }
        }
        
        $tests['total_streams'] = array(
            'status' => $stream_count > 0 ? 'pass' : 'warning',
            'message' => "Total streams configured: {$stream_count}",
            'value' => $stream_count
        );
        
        $this->log_debug('info', "Stream configuration test completed. Found {$stream_count} streams");
        
        return $tests;
    }
    
    private function calculate_system_health($results) {
        $total_tests = 0;
        $passed_tests = 0;
        
        foreach ($results as $category => $tests) {
            foreach ($tests as $test) {
                $total_tests++;
                if ($test['status'] === 'pass') {
                    $passed_tests++;
                }
            }
        }
        
        return $total_tests > 0 ? round(($passed_tests / $total_tests) * 100) : 0;
    }
    
    private function parse_memory_limit($memory_limit) {
        $memory_limit = trim($memory_limit);
        $last = strtolower($memory_limit[strlen($memory_limit)-1]);
        $memory_limit = (int) $memory_limit;
        
        switch($last) {
            case 'g':
                $memory_limit *= 1024;
            case 'm':
                $memory_limit *= 1024;
            case 'k':
                $memory_limit *= 1024;
        }
        
        return $memory_limit / (1024 * 1024); // Return in MB
    }
    
    public function get_debug_log() {
        check_ajax_referer('airwave_nonce', 'nonce');
        
        if (!current_user_can('manage_options')) {
            wp_send_json_error('Insufficient permissions');
        }
        
        wp_send_json_success(array(
            'log' => $this->get_debug_log_content()
        ));
    }
    
    public function clear_debug_log() {
        check_ajax_referer('airwave_nonce', 'nonce');
        
        if (!current_user_can('manage_options')) {
            wp_send_json_error('Insufficient permissions');
        }
        
        delete_option('airwave_debug_log');
        $this->log_debug('info', 'Debug log cleared');
        
        wp_send_json_success('Debug log cleared');
    }
    
    public function download_debug_log() {
        check_ajax_referer('airwave_nonce', 'nonce');
        
        if (!current_user_can('manage_options')) {
            wp_send_json_error('Insufficient permissions');
        }
        
        $log_content = $this->get_debug_log_content();
        $filename = 'airwave-debug-log-' . date('Y-m-d-H-i-s') . '.txt';
        
        header('Content-Type: text/plain');
        header('Content-Disposition: attachment; filename="' . $filename . '"');
        header('Content-Length: ' . strlen($log_content));
        
        echo $log_content;
        exit;
    }
    
    private function log_debug($level, $message) {
        $debug_log = get_option('airwave_debug_log', array());
        
        $entry = array(
            'timestamp' => current_time('Y-m-d H:i:s'),
            'level' => $level,
            'message' => $message,
            'user_id' => get_current_user_id()
        );
        
        $debug_log[] = $entry;
        
        // Keep only last 1000 entries
        if (count($debug_log) > 1000) {
            $debug_log = array_slice($debug_log, -1000);
        }
        
        update_option('airwave_debug_log', $debug_log);
    }
    
    private function get_debug_log_content() {
        $debug_log = get_option('airwave_debug_log', array());
        $content = '';
        
        foreach ($debug_log as $entry) {
            $content .= "[{$entry['timestamp']}] [{$entry['level']}] {$entry['message']}\n";
        }
        
        return $content ?: "No debug entries found.\n";
    }
    
    private function refresh_matrix_connection($login_data) {
        if (empty($login_data) || empty($login_data['username']) || empty($login_data['homeserver'])) {
            return array('success' => false, 'error' => 'Missing login credentials');
        }
        
        // Try to re-authenticate with stored username and homeserver
        // Note: We don't have the password stored, so we'll need to prompt for re-login
        return array('success' => false, 'error' => 'Please re-login - password not stored for security');
    }

    public function apply_test_credentials() {
        check_ajax_referer('airwave_nonce', 'nonce');

        if (!current_user_can('manage_options')) {
            wp_send_json_error('Insufficient permissions');
        }

        if (!function_exists('airwave_has_local_test_credentials') || !airwave_has_local_test_credentials()) {
            wp_send_json_error('Local credentials file not found');
        }

        $creds = airwave_load_local_test_credentials();
        if (!is_array($creds) || empty($creds)) {
            wp_send_json_error('Failed to load credentials');
        }

        $settings = get_option('airwave_settings', array());
        $updates = array();

        if (isset($creds['matrix']) && is_array($creds['matrix'])) {
            $m = $creds['matrix'];
            $updates['matrix_homeserver'] = isset($m['homeserver']) ? $m['homeserver'] : ($m['homeserver_url'] ?? '');
            $updates['matrix_custom_homeserver'] = $m['custom_homeserver'] ?? '';
            $updates['matrix_username'] = $m['username'] ?? '';
            $updates['matrix_password'] = $m['password'] ?? '';
            $updates['matrix_room'] = $m['room'] ?? ($m['room_id'] ?? '');
        }

        if (isset($creds['sonicpanel']) && is_array($creds['sonicpanel'])) {
            $s = $creds['sonicpanel'];
            $updates['sonic_enabled'] = 1;
            $updates['sonic_server_url'] = $s['server_url'] ?? '';
            $updates['sonic_port'] = isset($s['port']) ? preg_replace('/[^0-9]/', '', (string)$s['port']) : ($settings['sonic_port'] ?? '');
            $updates['sonic_stream_url'] = $s['stream_url'] ?? '';
            $updates['sonic_json_api'] = $s['json_api'] ?? '';
            $updates['sonic_admin_url'] = $s['admin_url'] ?? '';
            $updates['sonic_username'] = $s['username'] ?? '';
            $updates['sonic_password'] = $s['password'] ?? '';
            $updates['sonic_widget_script'] = $s['widget_script'] ?? '';
        }

        $input = array_merge($settings, $updates);
        $sanitized = $this->sanitize_settings($input);
        update_option('airwave_settings', $sanitized);

        wp_send_json_success(array(
            'message' => 'Test credentials applied',
            'applied' => array_keys($updates)
        ));
    }

    private function render_developer_tab($settings) {
        $has = function_exists('airwave_has_local_test_credentials') ? airwave_has_local_test_credentials() : false;
        $file = function_exists('airwave_test_credentials_file') ? airwave_test_credentials_file() : '';
        $creds = $has && function_exists('airwave_load_local_test_credentials') ? airwave_load_local_test_credentials() : array();
        ?>
        <div class="airwave-tab-content">
            <h3>Developer Tools</h3>
            <div class="airwave-info-notice" style="background: #f0f8ff; border: 1px solid #0073aa; border-radius: 4px; padding: 15px; margin-bottom: 20px;">
                <p>Load local test credentials from <code>TEST-CREDENTIALS-LOCAL.json</code> and apply them to plugin settings. This file is excluded from packaging and ignored by Git.</p>
            </div>

            <p><strong>Credentials file:</strong> <?php echo $has ? '<span style="color:green">Found</span>' : '<span style="color:red">Not found</span>'; ?></p>
            <?php if ($has): ?>
                <p><strong>Path:</strong> <?php echo esc_html($file); ?></p>
                <p>
                    <button type="button" class="button button-primary" id="airwave-apply-test-credentials">Apply Test Credentials</button>
                    <span id="airwave-apply-test-credentials-status" style="margin-left:10px;"></span>
                </p>
                <details>
                    <summary>Preview loaded credentials</summary>
                    <pre style="max-height:260px; overflow:auto;"><?php echo esc_html(json_encode($creds, JSON_PRETTY_PRINT)); ?></pre>
                </details>
            <?php else: ?>
                <p>Place <code>TEST-CREDENTIALS-LOCAL.json</code> in your plugin root or the WordPress webroot.</p>
            <?php endif; ?>
        </div>

        <script>
        jQuery(function($){
            $('#airwave-apply-test-credentials').on('click', function(){
                var $btn = $(this);
                var $status = $('#airwave-apply-test-credentials-status');
                $btn.prop('disabled', true);
                $status.text('Applying...');
                $.post(airwaveAjax.ajaxUrl, {
                    action: 'airwave_apply_test_credentials',
                    nonce: airwaveAjax.nonce
                }).done(function(resp){
                    if (resp && resp.success) {
                        $status.text('Applied successfully.').css('color','green');
                    } else {
                        var msg = (resp && resp.data) ? resp.data : 'Failed to apply credentials.';
                        $status.text(msg).css('color','red');
                    }
                }).fail(function(){
                    $status.text('Request failed.').css('color','red');
                }).always(function(){
                    $btn.prop('disabled', false);
                });
            });
        });
        </script>
        <?php
    }
}