<?php
/**
 * WordPress Cache Clear Script for Airwave Audio Player Matrix (includes)
 * Run this script to clear all WordPress caches and force version refresh
 */

// Start output buffering and set headers immediately to prevent header warnings
if (php_sapi_name() !== 'cli') {
    ob_start();
    if (!headers_sent()) {
        header('Cache-Control: no-cache, no-store, must-revalidate');
        header('Pragma: no-cache');
        header('Expires: 0');
    }
}

// Attempt to load WordPress if not already in context
if (!defined('ABSPATH')) {
    $wp_load_paths = [
        dirname(__FILE__, 4) . '/wp-load.php',
        dirname(__FILE__, 5) . '/wp-load.php',
        __DIR__ . '/../../../wp-load.php',
        __DIR__ . '/../../../../wp-load.php',
        __DIR__ . '/../../../../../wp-load.php',
    ];

    $wp_loaded = false;
    foreach ($wp_load_paths as $path) {
        if (file_exists($path)) {
            require_once $path;
            $wp_loaded = true;
            break;
        }
    }

    if (!$wp_loaded) {
        die('WordPress not found. Please run this script from within the WordPress installation.');
    }
}

// Security: allow only logged-in admins when running via web
if (php_sapi_name() !== 'cli') {
    if (!is_user_logged_in() || !current_user_can('manage_options')) {
        wp_die('Unauthorized. Please log in as an administrator.');
    }
    // Additional nocache headers if function exists
    if (function_exists('nocache_headers')) { 
        nocache_headers(); 
    }
}

// Clear WordPress object cache
if (function_exists('wp_cache_flush')) {
    wp_cache_flush();
    echo "✅ Object cache flushed<br>";
}

// Clear WordPress transients
global $wpdb;
$transient_count = $wpdb->query("DELETE FROM {$wpdb->options} WHERE option_name LIKE '_transient_%'");
$site_transient_count = $wpdb->query("DELETE FROM {$wpdb->options} WHERE option_name LIKE '_site_transient_%'");
echo "✅ Cleared {$transient_count} transients and {$site_transient_count} site transients<br>";

// Clear WordPress rewrite rules
if (function_exists('flush_rewrite_rules')) {
    flush_rewrite_rules();
    echo "✅ Rewrite rules flushed<br>";
}

// Clear WordPress plugin cache
if (function_exists('wp_clean_plugins_cache')) {
    wp_clean_plugins_cache();
    echo "✅ Plugin cache cleaned<br>";
}

// Clear WordPress update cache
if (function_exists('wp_clean_update_cache')) {
    wp_clean_update_cache();
    echo "✅ Update cache cleaned<br>";
}

// Clear any custom plugin caches
delete_option('airwave_plugin_version_cache');
delete_transient('airwave_plugin_info');
delete_option('airwave_last_version_check');
echo "✅ Custom plugin caches cleared<br>";

// Force WordPress to re-read plugin headers and show version
$plugin_file = plugin_dir_path(__FILE__) . '../airwave-audio-player-matrix.php';
$version_text = '';
if (file_exists($plugin_file) && function_exists('get_plugin_data')) {
    $plugin_data = get_plugin_data($plugin_file);
    $version_text = isset($plugin_data['Version']) ? $plugin_data['Version'] : '';
    echo "✅ Plugin data refreshed - Version: " . esc_html($version_text) . "<br>";
} else {
    echo "⚠️ Plugin file not found at: " . esc_html($plugin_file) . "<br>";
}

// Final output

echo "<br><strong>🎉 WordPress caches cleared successfully!</strong><br>";
if ($version_text) {
    echo "Plugin version is <strong>" . esc_html($version_text) . "</strong> in the WordPress admin.<br>";
}

echo "<br><a href='" . esc_url(admin_url('admin.php?page=airwave-dashboard')) . "'>Go to Airwave Dashboard</a><br>";
echo "<a href='" . esc_url(admin_url('admin.php?page=airwave-settings&tab=matrix')) . "'>Go to Matrix Configuration</a><br>";

if (php_sapi_name() !== 'cli') { ob_end_flush(); }