<?php
if (!defined('ABSPATH')) { exit; }

class AirwaveMetaProxy {

    public function get_metadata($type, $base_url, $meta_url, $sonic_port) {
        $type = $type ? strtolower(trim($type)) : '';

        // Infer SonicPanel from hints if missing
        if (!$type) {
            if ($this->looks_like_sonicpanel($base_url) || $this->looks_like_sonicpanel($meta_url)) {
                $type = 'sonicpanel';
            }
        }

        // Build meta_url for SonicPanel if needed - always use HTTPS to avoid mixed content
        if ($type === 'sonicpanel' && !$meta_url && $base_url && $sonic_port) {
            $host = parse_url($base_url, PHP_URL_HOST);
            if ($host) {
                $meta_url = sprintf('https://%s/cp/get_info.php?p=%s', $host, $sonic_port);
            }
        }
        
        // Force HTTPS for get_info.php URLs to prevent mixed content issues
        if (!empty($meta_url) && strpos($meta_url, '/cp/get_info.php') !== false) {
            $meta_url = preg_replace('/^http:\/\//i', 'https://', $meta_url);
        }

        // Dispatch by type
        switch ($type) {
            case 'sonicpanel':
                return $this->sonicpanel($meta_url ?: '', $base_url ?: '', $sonic_port ?: '');
            case 'icecast':
            case 'shoutcast':
            default:
                // Minimal fallback (can be expanded later if needed)
                return $this->empty_payload('Unsupported or unknown type: ' . ($type ?: 'none'));
        }
    }

    private function looks_like_sonicpanel($url) {
        if (!$url) return false;
        return (bool) preg_match('~(instainternet|/cp/get_info\.php|/sonic|/sp/|sonicpanel)~i', $url);
    }

    private function sonicpanel($meta_url, $base_url, $sonic_port) {
        // Try to construct a sensible URL if missing - always use HTTPS
        if (!$meta_url && $base_url && $sonic_port) {
            $host = parse_url($base_url, PHP_URL_HOST);
            if ($host) {
                $meta_url = sprintf('https://%s/cp/get_info.php?p=%s', $host, $sonic_port);
            }
        }

        if (!$meta_url) {
            return $this->empty_payload('Missing meta_url for SonicPanel');
        }

        // Force HTTPS for get_info.php URLs to prevent mixed content issues
        if (strpos($meta_url, '/cp/get_info.php') !== false) {
            $meta_url = preg_replace('/^http:\/\//i', 'https://', $meta_url);
        }

        $json = $this->fetch_json($meta_url);
        if (!is_array($json)) {
            return $this->empty_payload('Invalid JSON from SonicPanel');
        }

        return $this->normalize_sonicpanel($json, $meta_url);
    }

    private function fetch_json($url) {
        // Ensure HTTPS for get_info.php to avoid mixed content
        if (strpos($url, '/cp/get_info.php') !== false) {
            $url = preg_replace('/^http:\/\//i', 'https://', $url);
        }
        
        $resp = wp_remote_get($url, array(
            'timeout'      => 10,
            'redirection'  => 5,
            'headers'      => array(
                'Accept' => 'application/json',
                'User-Agent' => 'Airwave Audio Player Matrix/' . (defined('AIRWAVE_PLUGIN_VERSION') ? AIRWAVE_PLUGIN_VERSION : '0.0.61')
            ),
            'sslverify'    => true,
        ));

        if (is_wp_error($resp)) {
            error_log('Airwave Meta Proxy: wp_remote_get error: ' . $resp->get_error_message() . ' for URL: ' . $url);
            return null;
        }
        $code = wp_remote_retrieve_response_code($resp);
        if ($code < 200 || $code >= 300) {
            error_log('Airwave Meta Proxy: HTTP ' . $code . ' for URL: ' . $url);
            return null;
        }
        $body = wp_remote_retrieve_body($resp);
        $data = json_decode($body, true);
        if (json_last_error() !== JSON_ERROR_NONE) {
            error_log('Airwave Meta Proxy: JSON decode error: ' . json_last_error_msg() . ' for URL: ' . $url);
            return null;
        }
        return is_array($data) ? $data : null;
    }

    private function normalize_sonicpanel(array $d, $source_url) {
        // Listeners
        $listeners = null;
        foreach (['listeners', 'ulistener', 'currentlisteners', 'listener_count'] as $k) {
            if (isset($d[$k])) { $listeners = (int) $d[$k]; break; }
        }

        // DJ - handle "No DJ" or empty strings
        $dj = '';
        foreach (['djusername', 'djname', 'currentsongdj', 'dj'] as $k) {
            if (!empty($d[$k]) && strtolower(trim($d[$k])) !== 'no dj') { 
                $dj = trim((string) $d[$k]); 
                break; 
            }
        }

        // Title / Artist
        $raw = '';
        foreach (['title', 'song', 'now_playing', 'nowplaying'] as $k) {
            if (isset($d[$k])) { $raw = trim((string)$d[$k]); break; }
        }
        $artist = '';
        $title  = $raw;
        if (strpos($raw, ' - ') !== false) {
            [$artist, $title] = array_map('trim', explode(' - ', $raw, 2));
        }

        // Artwork
        $albumArt = '';
        foreach (['art', 'song_image', 'songimage', 'image', 'artwork', 'album_art', 'cover'] as $k) {
            if (!empty($d[$k])) { $albumArt = (string)$d[$k]; break; }
        }

        // History (optional)
        $history = [];
        if (!empty($d['history']) && is_array($d['history'])) {
            foreach ($d['history'] as $h) {
                $history[] = array(
                    'played_at' => isset($h['date']) ? (string)$h['date'] : (isset($h['time']) ? (string)$h['time'] : ''),
                    'title'     => isset($h['title']) ? (string)$h['title'] : (isset($h['song']) ? (string)$h['song'] : ''),
                );
            }
        }

        return array(
            'source'     => 'sonicpanel',
            'source_url' => $source_url,
            'title'      => $title,
            'artist'     => $artist,
            'dj'         => $dj,
            'listeners'  => $listeners,
            'albumArt'   => $albumArt,
            'history'    => $history,
        );
    }

    private function empty_payload($reason = '') {
        return array(
            'source'     => 'unknown',
            'title'      => '',
            'artist'     => '',
            'dj'         => '',
            'listeners'  => null,
            'albumArt'   => '',
            'history'    => array(),
            'error'      => $reason,
        );
    }
}
