<?php
/**
 * Plugin Update Checker
 * Handles checking for updates from custom repository
 */

if (!defined('ABSPATH')) exit;

class Airwave_Update_Checker {
    
    private $api_url = 'https://spunwebtechnology.com/airwaveautomator/plugin-updates/';
    private $plugin_slug = 'airwave-audio-player-matrix';
    private $plugin_file;
    private $version;
    
    public function __construct($plugin_file) {
        $this->plugin_file = $plugin_file;
        $this->version = defined('AIRWAVE_PLUGIN_VERSION') ? AIRWAVE_PLUGIN_VERSION : '0.0.94';
        
        // Hook into WordPress update system
        add_filter('pre_set_site_transient_update_plugins', array($this, 'check_for_updates'));
        add_filter('plugins_api', array($this, 'plugin_information'), 10, 3);
        add_filter('upgrader_post_install', array($this, 'post_install'), 10, 3);
    }
    
    /**
     * Check for plugin updates
     */
    public function check_for_updates($transient) {
        if (empty($transient->checked)) {
            return $transient;
        }
        
        // Check if we've already checked recently (every 12 hours)
        $cache_key = 'airwave_update_check_' . $this->version;
        $cached = get_transient($cache_key);
        
        if ($cached !== false) {
            if (isset($cached->response[$this->plugin_file])) {
                $transient->response[$this->plugin_file] = $cached->response[$this->plugin_file];
            }
            return $transient;
        }
        
        // Fetch update info from repository
        $update_info = $this->fetch_update_info();
        
        if ($update_info && version_compare($this->version, $update_info['version'], '<')) {
            $transient->response[$this->plugin_file] = (object) array(
                'slug' => $this->plugin_slug,
                'plugin' => $this->plugin_file,
                'new_version' => $update_info['version'],
                'url' => $update_info['homepage'],
                'package' => $update_info['download_url'],
                'tested' => $update_info['tested'],
                'requires' => $update_info['requires'],
                'icons' => isset($update_info['icons']) ? $update_info['icons'] : array(),
                'banners' => isset($update_info['banners']) ? $update_info['banners'] : array(),
            );
            
            // Cache the response for 12 hours
            set_transient($cache_key, $transient, 12 * HOUR_IN_SECONDS);
        } else {
            // Cache that no update is available (12 hours)
            $transient->no_update[$this->plugin_file] = true;
            set_transient($cache_key, $transient, 12 * HOUR_IN_SECONDS);
        }
        
        return $transient;
    }
    
    /**
     * Fetch update information from repository
     */
    private function fetch_update_info() {
        $api_url = $this->api_url . 'info.json';
        
        $response = wp_remote_get($api_url, array(
            'timeout' => 15,
            'sslverify' => true,
            'headers' => array(
                'Accept' => 'application/json',
            ),
        ));
        
        if (is_wp_error($response)) {
            return false;
        }
        
        $body = wp_remote_retrieve_body($response);
        $data = json_decode($body, true);
        
        if (json_last_error() !== JSON_ERROR_NONE || !is_array($data)) {
            return false;
        }
        
        return $data;
    }
    
    /**
     * Provide plugin information for update details
     */
    public function plugin_information($result, $action, $args) {
        if ($action !== 'plugin_information' || $args->slug !== $this->plugin_slug) {
            return $result;
        }
        
        $update_info = $this->fetch_update_info();
        
        if (!$update_info) {
            return $result;
        }
        
        $info = new stdClass();
        $info->name = isset($update_info['name']) ? $update_info['name'] : 'Airwave Audio Player Matrix';
        $info->slug = $this->plugin_slug;
        $info->version = $update_info['version'];
        $info->author = isset($update_info['author']) ? $update_info['author'] : 'Spun Web Technology';
        $info->homepage = isset($update_info['homepage']) ? $update_info['homepage'] : 'https://spunwebtechnology.com/airwaveautomator/';
        $info->download_link = $update_info['download_url'];
        $info->requires = isset($update_info['requires']) ? $update_info['requires'] : '5.0';
        $info->tested = isset($update_info['tested']) ? $update_info['tested'] : get_bloginfo('version');
        $info->last_updated = isset($update_info['last_updated']) ? $update_info['last_updated'] : date('Y-m-d');
        $info->sections = array(
            'description' => isset($update_info['description']) ? $update_info['description'] : '',
            'changelog' => isset($update_info['changelog']) ? $update_info['changelog'] : '',
        );
        
        return $info;
    }
    
    /**
     * Post-installation hook
     */
    public function post_install($true, $hook_extra, $result) {
        global $wp_filesystem;
        
        if (isset($hook_extra['plugin']) && $hook_extra['plugin'] === $this->plugin_file) {
            $install_directory = plugin_dir_path($this->plugin_file);
            $wp_filesystem->move($result['destination'], $install_directory);
            $result['destination'] = $install_directory;
        }
        
        return $result;
    }
}

