<?php
/**
 * Widgets for Airwave Audio Player Matrix
 */

if (!defined('ABSPATH')) {
    exit;
}

class AirwaveRadioWidget extends WP_Widget {
    
    public function __construct() {
        parent::__construct(
            'airwave_radio_widget',
            'Airwave Radio Player',
            array('description' => 'Display radio streaming player with current song metadata')
        );
    }
    
    public function widget($args, $instance) {
        echo $args['before_widget'];
        
        if (!empty($instance['title'])) {
            echo $args['before_title'] . apply_filters('widget_title', $instance['title']) . $args['after_title'];
        }
        
        $stream_index = !empty($instance['stream_index']) ? intval($instance['stream_index']) : 0;
        $show_history = !empty($instance['show_history']) ? true : false;
        $show_visualizer = !empty($instance['show_visualizer']) ? true : false;
        
        ?>
        <div class="airwave-radio-widget" data-stream="<?php echo esc_attr($stream_index); ?>" data-show-history="<?php echo $show_history ? 'true' : 'false'; ?>" data-show-visualizer="<?php echo $show_visualizer ? 'true' : 'false'; ?>">
            <div class="airwave-player-controls">
                <div class="airwave-stream-select">
                    <select class="airwave-stream-dropdown">
                        <option value="">Select Stream...</option>
                    </select>
                </div>
                
                <div class="airwave-control-buttons">
                    <button class="airwave-play-btn">Play</button>
                    <button class="airwave-stop-btn" disabled>Stop</button>
                </div>
                
                <div class="airwave-volume-control">
                    <label>Volume</label>
                    <input type="range" class="airwave-volume-slider" min="0" max="1" step="0.05" value="0.8" />
                </div>
            </div>
            
            <?php if ($show_visualizer): ?>
            <div class="airwave-visualizer">
                <canvas class="airwave-viz-canvas" width="400" height="120"></canvas>
            </div>
            <?php endif; ?>
            
            <div class="airwave-now-playing">
                <div class="airwave-song-info">
                    <div class="airwave-song-title">-</div>
                    <div class="airwave-song-artist">-</div>
                </div>
                <div class="airwave-song-art">
                    <img class="airwave-album-art" src="" alt="" style="display: none;" />
                </div>
            </div>
            
            <?php if ($show_history): ?>
            <div class="airwave-song-history">
                <h4>Recent Songs</h4>
                <div class="airwave-history-list"></div>
            </div>
            <?php endif; ?>
            
            <audio class="airwave-audio-player" preload="none"></audio>
        </div>
        <?php
        
        echo $args['after_widget'];
    }
    
    public function form($instance) {
        $title = !empty($instance['title']) ? $instance['title'] : '';
        $stream_index = !empty($instance['stream_index']) ? $instance['stream_index'] : 0;
        $show_history = !empty($instance['show_history']) ? true : false;
        $show_visualizer = !empty($instance['show_visualizer']) ? true : false;
        
        ?>
        <p>
            <label for="<?php echo $this->get_field_id('title'); ?>">Title:</label>
            <input class="widefat" id="<?php echo $this->get_field_id('title'); ?>" name="<?php echo $this->get_field_name('title'); ?>" type="text" value="<?php echo esc_attr($title); ?>">
        </p>
        
        <p>
            <label for="<?php echo $this->get_field_id('stream_index'); ?>">Default Stream:</label>
            <select class="widefat" id="<?php echo $this->get_field_id('stream_index'); ?>" name="<?php echo $this->get_field_name('stream_index'); ?>">
                <?php
                $streams = get_option('airwave_streams', array());
                foreach ($streams as $index => $stream) {
                    echo '<option value="' . $index . '" ' . selected($stream_index, $index, false) . '>' . esc_html($stream['name']) . '</option>';
                }
                ?>
            </select>
        </p>
        
        <p>
            <input class="checkbox" type="checkbox" <?php checked($show_history); ?> id="<?php echo $this->get_field_id('show_history'); ?>" name="<?php echo $this->get_field_name('show_history'); ?>" />
            <label for="<?php echo $this->get_field_id('show_history'); ?>">Show Song History</label>
        </p>
        
        <p>
            <input class="checkbox" type="checkbox" <?php checked($show_visualizer); ?> id="<?php echo $this->get_field_id('show_visualizer'); ?>" name="<?php echo $this->get_field_name('show_visualizer'); ?>" />
            <label for="<?php echo $this->get_field_id('show_visualizer'); ?>">Show Audio Visualizer</label>
        </p>
        <?php
    }
    
    public function update($new_instance, $old_instance) {
        $instance = array();
        $instance['title'] = (!empty($new_instance['title'])) ? strip_tags($new_instance['title']) : '';
        $instance['stream_index'] = (!empty($new_instance['stream_index'])) ? intval($new_instance['stream_index']) : 0;
        $instance['show_history'] = !empty($new_instance['show_history']) ? true : false;
        $instance['show_visualizer'] = !empty($new_instance['show_visualizer']) ? true : false;
        
        return $instance;
    }
}

class AirwaveMatrixWidget extends WP_Widget {
    
    public function __construct() {
        parent::__construct(
            'airwave_matrix_widget',
            'Airwave Matrix Chat',
            array('description' => 'Display Matrix chat interface')
        );
    }
    
    public function widget($args, $instance) {
        echo $args['before_widget'];
        
        if (!empty($instance['title'])) {
            echo $args['before_title'] . apply_filters('widget_title', $instance['title']) . $args['after_title'];
        }
        
        $room_id = !empty($instance['room_id']) ? $instance['room_id'] : '';
        $show_room_selector = !empty($instance['show_room_selector']) ? true : false;
        
        ?>
        <div class="airwave-matrix-widget" data-room="<?php echo esc_attr($room_id); ?>" data-show-selector="<?php echo $show_room_selector ? 'true' : 'false'; ?>">
            <div class="airwave-matrix-status"></div>
            
            <?php if ($show_room_selector): ?>
            <div class="airwave-room-selector">
                <h4>Rooms</h4>
                <div class="airwave-room-list"></div>
                <div class="airwave-join-room">
                    <input type="text" class="airwave-room-input" placeholder="Enter room ID (e.g., #general:matrix.org)" />
                    <button class="airwave-join-btn">Join Room</button>
                </div>
            </div>
            <?php endif; ?>
            
            <div class="airwave-chat-container">
                <div class="airwave-current-room">
                    <span class="airwave-room-name">No room selected</span>
                </div>
                
                <div class="airwave-messages-container">
                    <div class="airwave-messages-list"></div>
                </div>
                
                <div class="airwave-message-input">
                    <input type="text" class="airwave-message-field" placeholder="Type your message..." />
                    <button class="airwave-send-btn">Send</button>
                </div>
            </div>
        </div>
        <?php
        
        echo $args['after_widget'];
    }
    
    public function form($instance) {
        $title = !empty($instance['title']) ? $instance['title'] : '';
        $room_id = !empty($instance['room_id']) ? $instance['room_id'] : '';
        $show_room_selector = !empty($instance['show_room_selector']) ? true : false;
        
        ?>
        <p>
            <label for="<?php echo $this->get_field_id('title'); ?>">Title:</label>
            <input class="widefat" id="<?php echo $this->get_field_id('title'); ?>" name="<?php echo $this->get_field_name('title'); ?>" type="text" value="<?php echo esc_attr($title); ?>">
        </p>
        
        <p>
            <label for="<?php echo $this->get_field_id('room_id'); ?>">Default Room ID:</label>
            <input class="widefat" id="<?php echo $this->get_field_id('room_id'); ?>" name="<?php echo $this->get_field_name('room_id'); ?>" type="text" value="<?php echo esc_attr($room_id); ?>" placeholder="#general:matrix.org">
        </p>
        
        <p>
            <input class="checkbox" type="checkbox" <?php checked($show_room_selector); ?> id="<?php echo $this->get_field_id('show_room_selector'); ?>" name="<?php echo $this->get_field_name('show_room_selector'); ?>" />
            <label for="<?php echo $this->get_field_id('show_room_selector'); ?>">Show Room Selector</label>
        </p>
        <?php
    }
    
    public function update($new_instance, $old_instance) {
        $instance = array();
        $instance['title'] = (!empty($new_instance['title'])) ? strip_tags($new_instance['title']) : '';
        $instance['room_id'] = (!empty($new_instance['room_id'])) ? strip_tags($new_instance['room_id']) : '';
        $instance['show_room_selector'] = !empty($new_instance['show_room_selector']) ? true : false;
        
        return $instance;
    }
}

class AirwaveCombinedWidget extends WP_Widget {
    
    public function __construct() {
        parent::__construct(
            'airwave_combined_widget',
            'Airwave Radio + Matrix Chat',
            array('description' => 'Display radio player and Matrix chat in tabs')
        );
    }
    
    public function widget($args, $instance) {
        echo $args['before_widget'];
        
        if (!empty($instance['title'])) {
            echo $args['before_title'] . apply_filters('widget_title', $instance['title']) . $args['after_title'];
        }
        
        ?>
        <div class="airwave-combined-widget">
            <div class="airwave-tabs">
                <nav class="airwave-tab-nav">
                    <button type="button" class="airwave-tab-btn active" data-tab="radio">Radio</button>
                    <button type="button" class="airwave-tab-btn" data-tab="chat">Chat</button>
                </nav>
                
                <div class="airwave-tab-content">
                    <div id="radio" class="airwave-tab-panel active">
                        <div class="airwave-radio-content">
                            <div class="airwave-player-controls">
                                <div class="airwave-stream-select">
                                    <select class="airwave-stream-dropdown">
                                        <option value="">Select Stream...</option>
                                    </select>
                                </div>
                                
                                <div class="airwave-control-buttons">
                                    <button class="airwave-play-btn">Play</button>
                                    <button class="airwave-stop-btn" disabled>Stop</button>
                                </div>
                                
                                <div class="airwave-volume-control">
                                    <label>Volume</label>
                                    <input type="range" class="airwave-volume-slider" min="0" max="1" step="0.05" value="0.8" />
                                </div>
                            </div>
                            
                            <div class="airwave-now-playing">
                                <div class="airwave-song-info">
                                    <div class="airwave-song-title">-</div>
                                    <div class="airwave-song-artist">-</div>
                                </div>
                                <div class="airwave-song-art">
                                    <img class="airwave-album-art" src="" alt="" style="display: none;" />
                                </div>
                            </div>
                            
                            <audio class="airwave-audio-player" preload="none"></audio>
                        </div>
                    </div>
                    
                    <div id="chat" class="airwave-tab-panel">
                        <div class="airwave-matrix-content">
                            <div class="airwave-matrix-status"></div>
                            
                            <div class="airwave-room-selector">
                                <h4>Rooms</h4>
                                <div class="airwave-room-list"></div>
                                <div class="airwave-join-room">
                                    <input type="text" class="airwave-room-input" placeholder="Enter room ID (e.g., #general:matrix.org)" />
                                    <button class="airwave-join-btn">Join Room</button>
                                </div>
                            </div>
                            
                            <div class="airwave-chat-container">
                                <div class="airwave-current-room">
                                    <span class="airwave-room-name">No room selected</span>
                                </div>
                                
                                <div class="airwave-messages-container">
                                    <div class="airwave-messages-list"></div>
                                </div>
                                
                                <div class="airwave-message-input">
                                    <input type="text" class="airwave-message-field" placeholder="Type your message..." />
                                    <button class="airwave-send-btn">Send</button>
                                </div>
                            </div>
                        </div>
                    </div>
                </div>
            </div>
        </div>
        <?php
        
        echo $args['after_widget'];
    }
    
    public function form($instance) {
        $title = !empty($instance['title']) ? $instance['title'] : '';
        
        ?>
        <p>
            <label for="<?php echo $this->get_field_id('title'); ?>">Title:</label>
            <input class="widefat" id="<?php echo $this->get_field_id('title'); ?>" name="<?php echo $this->get_field_name('title'); ?>" type="text" value="<?php echo esc_attr($title); ?>">
        </p>
        <?php
    }
    
    public function update($new_instance, $old_instance) {
        $instance = array();
        $instance['title'] = (!empty($new_instance['title'])) ? strip_tags($new_instance['title']) : '';
        
        return $instance;
    }
}

class AirwaveSongHistoryWidget extends WP_Widget {
    
    public function __construct() {
        parent::__construct(
            'airwave_song_history_widget',
            'Airwave Song History',
            array('description' => 'Display current song and recent song history')
        );
    }
    
    public function widget($args, $instance) {
        echo $args['before_widget'];
        
        if (!empty($instance['title'])) {
            echo $args['before_title'] . apply_filters('widget_title', $instance['title']) . $args['after_title'];
        }
        
        $stream_index = !empty($instance['stream_index']) ? intval($instance['stream_index']) : 0;
        $history_limit = !empty($instance['history_limit']) ? intval($instance['history_limit']) : 10;
        
        ?>
        <div class="airwave-song-history-widget" data-stream="<?php echo esc_attr($stream_index); ?>" data-limit="<?php echo esc_attr($history_limit); ?>">
            <div class="airwave-current-song">
                <h4>Now Playing</h4>
                <div class="airwave-song-info">
                    <div class="airwave-song-title">-</div>
                    <div class="airwave-song-artist">-</div>
                </div>
                <div class="airwave-song-art">
                    <img class="airwave-album-art" src="" alt="" style="display: none;" />
                </div>
            </div>
            
            <div class="airwave-song-history">
                <h4>Recent Songs</h4>
                <div class="airwave-history-list"></div>
            </div>
        </div>
        <?php
        
        echo $args['after_widget'];
    }
    
    public function form($instance) {
        $title = !empty($instance['title']) ? $instance['title'] : '';
        $stream_index = !empty($instance['stream_index']) ? $instance['stream_index'] : 0;
        $history_limit = !empty($instance['history_limit']) ? $instance['history_limit'] : 10;
        
        ?>
        <p>
            <label for="<?php echo $this->get_field_id('title'); ?>">Title:</label>
            <input class="widefat" id="<?php echo $this->get_field_id('title'); ?>" name="<?php echo $this->get_field_name('title'); ?>" type="text" value="<?php echo esc_attr($title); ?>">
        </p>
        
        <p>
            <label for="<?php echo $this->get_field_id('stream_index'); ?>">Stream:</label>
            <select class="widefat" id="<?php echo $this->get_field_id('stream_index'); ?>" name="<?php echo $this->get_field_name('stream_index'); ?>">
                <?php
                $streams = get_option('airwave_streams', array());
                foreach ($streams as $index => $stream) {
                    echo '<option value="' . $index . '" ' . selected($stream_index, $index, false) . '>' . esc_html($stream['name']) . '</option>';
                }
                ?>
            </select>
        </p>
        
        <p>
            <label for="<?php echo $this->get_field_id('history_limit'); ?>">History Limit:</label>
            <input class="widefat" id="<?php echo $this->get_field_id('history_limit'); ?>" name="<?php echo $this->get_field_name('history_limit'); ?>" type="number" value="<?php echo esc_attr($history_limit); ?>" min="2" max="20">
        </p>
        <?php
    }
    
    public function update($new_instance, $old_instance) {
        $instance = array();
        $instance['title'] = (!empty($new_instance['title'])) ? strip_tags($new_instance['title']) : '';
        $instance['stream_index'] = (!empty($new_instance['stream_index'])) ? intval($new_instance['stream_index']) : 0;
        return $instance;
    }
}

// Dashboard Widget for Matrix Login
class AirwaveMatrixLoginWidget {
    
    public function __construct() {
        add_action('wp_dashboard_setup', array($this, 'add_dashboard_widget'));
        add_action('wp_ajax_airwave_matrix_login', array($this, 'handle_matrix_login'));
        add_action('wp_ajax_airwave_matrix_logout', array($this, 'handle_matrix_logout'));
    }
    
    public function add_dashboard_widget() {
        wp_add_dashboard_widget(
            'airwave_matrix_login_widget',
            'Airwave Matrix Chat Login',
            array($this, 'render_dashboard_widget'),
            null,
            null,
            'normal',
            'high'
        );
        
        // Force the widget to be visible by default
        $user_id = get_current_user_id();
        $meta_key = 'meta-box-order_dashboard';
        $meta_value = get_user_meta($user_id, $meta_key, true);
        
        if (empty($meta_value)) {
            $meta_value = array(
                'normal' => 'airwave_matrix_login_widget,',
                'side' => '',
                'advanced' => ''
            );
            update_user_meta($user_id, $meta_key, $meta_value);
        } else {
            if (!is_array($meta_value)) {
                $meta_value = array();
            }
            if (!isset($meta_value['normal']) || strpos($meta_value['normal'], 'airwave_matrix_login_widget') === false) {
                $meta_value['normal'] = 'airwave_matrix_login_widget,' . $meta_value['normal'];
                update_user_meta($user_id, $meta_key, $meta_value);
            }
        }
    }
    
    public function render_dashboard_widget() {
        $matrix_settings = get_option('airwave_matrix_settings', array());
        $is_logged_in = !empty($matrix_settings['user']) && !empty($matrix_settings['access_token']);
        
        ?>
        <div class="airwave-dashboard-widget">
            <div class="airwave-matrix-login-widget">
                <?php if ($is_logged_in): ?>
                    <h4>Matrix Chat Status</h4>
                    <div class="airwave-login-status-widget success">
                        <span class="dashicons dashicons-yes-alt"></span> 
                        Logged in as: <strong><?php echo esc_html($matrix_settings['user']); ?></strong>
                    </div>
                    <div class="airwave-login-actions-widget">
                        <button type="button" class="airwave-logout-btn-widget" onclick="airwaveMatrixLogout()">
                            <span class="dashicons dashicons-exit"></span> Logout
                        </button>
                        <button type="button" class="airwave-login-btn-widget" onclick="window.open('<?php echo home_url('/chat/'); ?>', '_blank')">
                            <span class="dashicons dashicons-external"></span> Go to Chat
                        </button>
                    </div>
                <?php else: ?>
                    <h4>Matrix Chat Login</h4>
                    <div class="airwave-login-form-widget">
                        <div class="airwave-login-field">
                            <label for="airwave-matrix-homeserver-widget">Homeserver:</label>
                            <select id="airwave-matrix-homeserver-widget" class="airwave-homeserver-select-widget">
                                <option value="https://matrix-client.matrix.org">matrix.org</option>
                                <option value="https://matrix.element.io">Element (matrix.org)</option>
                                <option value="custom">Custom Homeserver</option>
                            </select>
                            <input type="text" id="airwave-custom-homeserver-widget" class="airwave-custom-homeserver-widget" placeholder="https://your-homeserver.com" style="display: none;" />
                        </div>
                        
                        <div class="airwave-login-field">
                            <label for="airwave-matrix-username-widget">Username:</label>
                            <input type="text" id="airwave-matrix-username-widget" class="airwave-username-input-widget" placeholder="@username:domain.com" />
                        </div>
                        
                        <div class="airwave-login-field">
                            <label for="airwave-matrix-password-widget">Password:</label>
                            <input type="password" id="airwave-matrix-password-widget" class="airwave-password-input-widget" placeholder="Your Matrix password" />
                        </div>
                        
                        <div class="airwave-login-field">
                            <label for="airwave-matrix-display-name-widget">Display Name (optional):</label>
                            <input type="text" id="airwave-matrix-display-name-widget" class="airwave-display-name-input-widget" placeholder="Your display name" />
                        </div>
                        
                        <div class="airwave-login-actions-widget">
                            <button type="button" class="airwave-login-btn-widget" onclick="airwaveMatrixLogin()">
                                <span class="dashicons dashicons-admin-users"></span> Login to Chat
                            </button>
                        </div>
                        
                        <div class="airwave-login-status-widget" id="airwave-login-status-widget"></div>
                    </div>
                    
                    <div class="airwave-login-help">
                        <p><strong>Need a Matrix account?</strong></p>
                        <p>You can create a free account at <a href="https://app.element.io" target="_blank">Element</a> or <a href="https://matrix.org" target="_blank">matrix.org</a></p>
                    </div>
                <?php endif; ?>
            </div>
        </div>
        
        <script>
        function airwaveMatrixLogin() {
            var homeserver = document.getElementById('airwave-matrix-homeserver-widget').value;
            if (homeserver === 'custom') {
                homeserver = document.getElementById('airwave-custom-homeserver-widget').value;
            }
            
            var username = document.getElementById('airwave-matrix-username-widget').value.trim();
            var password = document.getElementById('airwave-matrix-password-widget').value;
            var displayName = document.getElementById('airwave-matrix-display-name-widget').value.trim();
            
            if (!homeserver || !username || !password) {
                showLoginStatus('Please fill in all required fields', 'error');
                return;
            }
            
            var statusDiv = document.getElementById('airwave-login-status-widget');
            statusDiv.innerHTML = '<span class="spinner is-active"></span> Logging in...';
            statusDiv.className = 'airwave-login-status-widget info';
            statusDiv.style.display = 'block';
            
            jQuery.ajax({
                url: ajaxurl,
                type: 'POST',
                data: {
                    action: 'airwave_matrix_login',
                    homeserver: homeserver,
                    username: username,
                    password: password,
                    display_name: displayName,
                    nonce: '<?php echo wp_create_nonce('airwave_matrix_login'); ?>'
                },
                success: function(response) {
                    if (response.success) {
                        showLoginStatus('Login successful! Refreshing...', 'success');
                        setTimeout(function() {
                            location.reload();
                        }, 1000);
                    } else {
                        showLoginStatus(response.data || 'Login failed', 'error');
                    }
                },
                error: function() {
                    showLoginStatus('Connection error', 'error');
                }
            });
        }
        
        function airwaveMatrixLogout() {
            if (!confirm('Are you sure you want to logout?')) {
                return;
            }
            
            jQuery.ajax({
                url: ajaxurl,
                type: 'POST',
                data: {
                    action: 'airwave_matrix_logout',
                    nonce: '<?php echo wp_create_nonce('airwave_matrix_logout'); ?>'
                },
                success: function(response) {
                    if (response.success) {
                        location.reload();
                    } else {
                        alert('Logout failed');
                    }
                },
                error: function() {
                    alert('Connection error');
                }
            });
        }
        
        function showLoginStatus(message, type) {
            var statusDiv = document.getElementById('airwave-login-status-widget');
            statusDiv.innerHTML = message;
            statusDiv.className = 'airwave-login-status-widget ' + type;
            statusDiv.style.display = 'block';
        }
        
        // Homeserver selection
        document.getElementById('airwave-matrix-homeserver-widget').addEventListener('change', function() {
            var customInput = document.getElementById('airwave-custom-homeserver-widget');
            if (this.value === 'custom') {
                customInput.style.display = 'block';
            } else {
                customInput.style.display = 'none';
            }
        });
        </script>
        <?php
    }
    
    public function handle_matrix_login() {
        check_ajax_referer('airwave_matrix_login', 'nonce');
        
        $homeserver = sanitize_url($_POST['homeserver'] ?? '');
        $username = sanitize_text_field($_POST['username'] ?? '');
        $password = isset($_POST['password']) ? $_POST['password'] : '';
        $display_name = sanitize_text_field($_POST['display_name'] ?? '');
        
        // Store login credentials
        $matrix_settings = get_option('airwave_matrix_settings', array());
        $matrix_settings['homeserver'] = $homeserver;
        $matrix_settings['user'] = $username;
        $matrix_settings['password'] = $password;
        $matrix_settings['display_name'] = $display_name;
        $matrix_settings['access_token'] = 'stored'; // Placeholder for now
        $matrix_settings['logged_in'] = true;
        
        update_option('airwave_matrix_settings', $matrix_settings);
        
        wp_send_json_success('Login successful');
    }
    
    public function handle_matrix_logout() {
        check_ajax_referer('airwave_matrix_logout', 'nonce');
        
        $matrix_settings = get_option('airwave_matrix_settings', array());
        $matrix_settings['user'] = '';
        $matrix_settings['password'] = '';
        $matrix_settings['access_token'] = '';
        $matrix_settings['logged_in'] = false;
        
        update_option('airwave_matrix_settings', $matrix_settings);
        
        wp_send_json_success('Logout successful');
    }
}

// Initialize dashboard widget
new AirwaveMatrixLoginWidget();
